package com.ifourthwall.kafka.config;

import com.ifourthwall.kafka.IFWKafkaClient;
import org.apache.commons.lang3.StringUtils;
import org.apache.kafka.clients.admin.NewTopic;
import org.apache.kafka.clients.producer.ProducerConfig;
import org.apache.kafka.common.serialization.StringSerializer;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.boot.autoconfigure.AutoConfigureOrder;
import org.springframework.boot.autoconfigure.condition.ConditionalOnProperty;
import org.springframework.boot.context.properties.EnableConfigurationProperties;
import org.springframework.context.annotation.Configuration;
import org.springframework.context.support.GenericApplicationContext;
import org.springframework.core.Ordered;
import org.springframework.core.annotation.Order;
import org.springframework.kafka.core.DefaultKafkaProducerFactory;
import org.springframework.kafka.core.KafkaTemplate;
import org.springframework.util.CollectionUtils;

import javax.annotation.PostConstruct;
import java.util.HashMap;
import java.util.Map;

/**
 * @Author: hao.dai
 * @Date: 2019/12/24 15:27
 * @Description:
 */
@Configuration
@ConditionalOnProperty(prefix = "ifw.kafka.producer", name = "enable", havingValue = "true", matchIfMissing = false)
@EnableConfigurationProperties(value = {IFWKafkaProducerProperty.class})
@AutoConfigureOrder(value = Ordered.HIGHEST_PRECEDENCE + 3000)
public class IFWKafkaProducerConfig {

    @Autowired
    private IFWKafkaProducerProperty ifwKafkaProducerProperty;

    @Autowired
    private GenericApplicationContext applicationContext;

    @PostConstruct
    public void initKafkaProducer() {
        if (!CollectionUtils.isEmpty(ifwKafkaProducerProperty.getConfig())) {
            ifwKafkaProducerProperty.getConfig()
                    .forEach(ifwKafkaProperties -> {
                        Map<String, Object> props = new HashMap<>();
                        props.put(ProducerConfig.BOOTSTRAP_SERVERS_CONFIG, ifwKafkaProperties.getBootstrapServers());
                        props.put(ProducerConfig.KEY_SERIALIZER_CLASS_CONFIG, StringSerializer.class);
                        props.put(ProducerConfig.VALUE_SERIALIZER_CLASS_CONFIG, StringSerializer.class);

                        DefaultKafkaProducerFactory defaultKafkaProducerFactory = new DefaultKafkaProducerFactory<>(props);

                        //手动注册  template
                        applicationContext.registerBean(StringUtils.joinWith("-", KafkaTemplate.class,
                                ifwKafkaProperties.getKafkaTopic()),
                                KafkaTemplate.class,
                                () -> new KafkaTemplate<String, Object>(defaultKafkaProducerFactory));

                        //手动注册 topic
                        applicationContext.registerBean(StringUtils.joinWith("-", NewTopic.class,
                                ifwKafkaProperties.getKafkaTopic()),
                                NewTopic.class,
                                () -> new NewTopic(ifwKafkaProperties.getKafkaTopic(),
                                        ifwKafkaProperties.getNumPartition(), ifwKafkaProperties.getReplicationFactor())


                        );

                        //手动注册 IFWKafkaClient
                        applicationContext.registerBean(StringUtils.joinWith("-", "IFWKafkaClient",
                                ifwKafkaProperties.getKafkaTopic()),
                                IFWKafkaClient.class,
                                () -> {
                                    String beanName = StringUtils.joinWith("-", KafkaTemplate.class,
                                            ifwKafkaProperties.getKafkaTopic());
                                    Object kafkaTemplate = applicationContext.getBean(beanName);
                                    if (kafkaTemplate == null) {
                                        throw new IllegalStateException("未找到name:" + beanName + "对应的Bean");
                                    }
                                    return new IFWKafkaClient((KafkaTemplate) kafkaTemplate, ifwKafkaProperties.getKafkaTopic());
                                }
                        );

                    });
        }
    }
}
