package com.ifourthwall.kafka.config;

import com.ifourthwall.kafka.DataHandler;
import com.ifourthwall.kafka.KafkaDataConsumer;
import org.apache.commons.lang3.StringUtils;
import org.apache.kafka.clients.consumer.ConsumerConfig;
import org.apache.kafka.common.serialization.StringDeserializer;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.boot.autoconfigure.AutoConfigureOrder;
import org.springframework.boot.autoconfigure.condition.ConditionalOnProperty;
import org.springframework.boot.context.properties.EnableConfigurationProperties;
import org.springframework.context.annotation.Configuration;
import org.springframework.context.support.GenericApplicationContext;
import org.springframework.core.Ordered;
import org.springframework.core.annotation.Order;
import org.springframework.kafka.config.ConcurrentKafkaListenerContainerFactory;
import org.springframework.kafka.core.DefaultKafkaConsumerFactory;
import org.springframework.kafka.listener.ConcurrentMessageListenerContainer;
import org.springframework.kafka.listener.ContainerProperties;
import org.springframework.util.CollectionUtils;

import javax.annotation.PostConstruct;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

/**
 * @Author: hao.dai
 * @Date: 2019/12/25 15:09
 * @Description:
 */
@Configuration
@ConditionalOnProperty(prefix = "ifw.kafka.consumer", name = "enable", havingValue = "true", matchIfMissing = false)
@EnableConfigurationProperties(value = {IFWKafkaConsumerProperty.class})
@AutoConfigureOrder(value = Ordered.HIGHEST_PRECEDENCE + 3000)
public class IFWKafkaConsumerConfig {

    @Autowired
    private IFWKafkaConsumerProperty ifwKafkaConsumerProperty;

    @Autowired
    private List<DataHandler> dataHandlerList;

    @Autowired
    private GenericApplicationContext applicationContext;

    @PostConstruct
    public void initKafkaConsumer() {
        if (!CollectionUtils.isEmpty(ifwKafkaConsumerProperty.getConfig())) {
            ifwKafkaConsumerProperty.getConfig().stream().forEach(consumerConfigProp -> {
                Map<String, Object> configs = new HashMap<String, Object>(); //参数
                configs.put(ConsumerConfig.BOOTSTRAP_SERVERS_CONFIG, consumerConfigProp.getBootstrapServers());
                configs.put(ConsumerConfig.GROUP_ID_CONFIG, consumerConfigProp.getConsumerGroupId());
                configs.put(ConsumerConfig.ENABLE_AUTO_COMMIT_CONFIG, consumerConfigProp.getConsumerEnableAutoCommit());
                configs.put(ConsumerConfig.AUTO_COMMIT_INTERVAL_MS_CONFIG, consumerConfigProp.getConsumerAutoCommitIntervalMs());
                configs.put(ConsumerConfig.SESSION_TIMEOUT_MS_CONFIG, consumerConfigProp.getConsumerSessionTimeoutMs());
                configs.put(ConsumerConfig.MAX_POLL_RECORDS_CONFIG, consumerConfigProp.getConsumerMaxPollRecords()); //批量消费数量
                configs.put(ConsumerConfig.AUTO_OFFSET_RESET_CONFIG, consumerConfigProp.getConsumerAutoOffsetReset());
                configs.put(ConsumerConfig.KEY_DESERIALIZER_CLASS_CONFIG, StringDeserializer.class);
                configs.put(ConsumerConfig.VALUE_DESERIALIZER_CLASS_CONFIG, StringDeserializer.class);

                DefaultKafkaConsumerFactory defaultKafkaConsumerFactory = new DefaultKafkaConsumerFactory<Object, Object>(configs);

                applicationContext.registerBean(StringUtils.joinWith("-", ConcurrentMessageListenerContainer.class,
                        consumerConfigProp.getKafkaTopic(), consumerConfigProp.getConsumerGroupId()),
                        ConcurrentMessageListenerContainer.class,
                        () -> {
                            ContainerProperties containerProps = new ContainerProperties(consumerConfigProp.getKafkaTopic());
                            containerProps.setGroupId(consumerConfigProp.getConsumerGroupId());
                            containerProps.setMessageListener(new KafkaDataConsumer(dataHandlerList));
                            ConcurrentKafkaListenerContainerFactory<Integer, String> factory = new ConcurrentKafkaListenerContainerFactory<>();
                            factory.setConsumerFactory(defaultKafkaConsumerFactory);
                            ConcurrentMessageListenerContainer<Integer, String> container =
                                    new ConcurrentMessageListenerContainer<Integer, String>(defaultKafkaConsumerFactory, containerProps);
                            container.setConcurrency(consumerConfigProp.getConcurrency());
                            return container;
                        });
            });
        }
    }

}
