package com.ifourthwall.config;

import com.alibaba.nacos.api.NacosFactory;
import com.alibaba.nacos.api.PropertyKeyConst;
import com.alibaba.nacos.api.config.ConfigService;
import com.alibaba.nacos.api.exception.NacosException;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.boot.SpringApplication;
import org.springframework.boot.env.EnvironmentPostProcessor;
import org.springframework.context.ApplicationContextInitializer;
import org.springframework.context.ConfigurableApplicationContext;
import org.springframework.core.Ordered;
import org.springframework.core.env.*;

import java.io.IOException;
import java.io.StringReader;
import java.util.Properties;


/**
 * @Author: hao.dai
 * @Project: ifourthwall-parent
 * @Date: 2020/4/10 11:46
 * @Description:
 * @Godzilla.tech
 */
@Slf4j
public class IFWApplicationContextInitializer implements
        ApplicationContextInitializer<ConfigurableApplicationContext>, EnvironmentPostProcessor, Ordered {
    public static final int DEFAULT_ORDER = 0;
    public static final String PROPERTY_SOURCE_NAME = "configCenterComposite";

    @Override
    public void initialize(ConfigurableApplicationContext applicationContext) {
        log.info("handle initialize environment........");
        initEnvironment(applicationContext.getEnvironment());
    }

    private void initEnvironment(ConfigurableEnvironment environment) {
        String enable = environment.getProperty("nacos.config.bootstrap.enable");
        if (enable == null || !StringUtils.equals("true", enable)) {
            return;
        }

        if (environment.getPropertySources().contains(PROPERTY_SOURCE_NAME)) {
            //already initialized
            return;
        }

        CompositePropertySource source = new CompositePropertySource(PROPERTY_SOURCE_NAME);
        //timeout 2s
        String configContent = acquireConfigContent(environment);
        if (StringUtils.isBlank(configContent)) {
            return;
        }
        Properties properties = new Properties();
        try {
            properties.load(new StringReader(configContent));
        } catch (IOException e) {
            throw new RuntimeException(e);
        }
        PropertySource propertySource = new PropertiesPropertySource("configCenter", properties);

        source.addPropertySource(propertySource);
        environment.getPropertySources().addFirst(source);
    }

    @Override
    public int getOrder() {
        return Ordered.HIGHEST_PRECEDENCE;
    }

    private String acquireConfigContent(ConfigurableEnvironment environment) {
        String dataId = environment.getProperty("nacos.config.data-id");
        String group = environment.getProperty("nacos.config.group");
        String type = environment.getProperty("nacos.config.type");
        String endpoint = environment.getProperty("nacos.config.endpoint");
        String namespace = environment.getProperty("nacos.config.namespace");
        String accessKey = environment.getProperty("nacos.config.access-key");
        String secretKey = environment.getProperty("nacos.config.secret-key");

        if (!"properties".equals(type)) {
            throw new IllegalArgumentException("Not support type. Supported type include [properties].");
        }

        Properties properties = new Properties();
        properties.put(PropertyKeyConst.ENDPOINT, endpoint);
        properties.put(PropertyKeyConst.NAMESPACE, namespace);
        properties.put(PropertyKeyConst.ACCESS_KEY, accessKey);
        properties.put(PropertyKeyConst.SECRET_KEY, secretKey);

        try {
            ConfigService configService = NacosFactory.createConfigService(properties);
            return configService.getConfig(dataId, group, 5000);
        } catch (NacosException e) {
            throw new RuntimeException(e);
        }
    }

    @Override
    public void postProcessEnvironment(ConfigurableEnvironment environment, SpringApplication application) {
        log.info("handle postProcessEnvironment........");
        initEnvironment(environment);
    }
}
