/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.github.zengfr.easymodbus4j.codec.tcp;

import io.netty.buffer.ByteBuf;
import io.netty.channel.ChannelHandlerContext;
import io.netty.handler.codec.ByteToMessageDecoder;
import io.netty.util.internal.logging.InternalLogger;
import io.netty.util.internal.logging.InternalLoggerFactory;
import java.util.List;

import com.github.zengfr.easymodbus4j.ModbusConsts;
import com.github.zengfr.easymodbus4j.func.request.ReadCoilsRequest;
import com.github.zengfr.easymodbus4j.func.request.ReadDiscreteInputsRequest;
import com.github.zengfr.easymodbus4j.func.request.ReadHoldingRegistersRequest;
import com.github.zengfr.easymodbus4j.func.request.ReadInputRegistersRequest;
import com.github.zengfr.easymodbus4j.func.request.WriteMultipleCoilsRequest;
import com.github.zengfr.easymodbus4j.func.request.WriteMultipleRegistersRequest;
import com.github.zengfr.easymodbus4j.func.request.WriteSingleCoilRequest;
import com.github.zengfr.easymodbus4j.func.request.WriteSingleRegisterRequest;
import com.github.zengfr.easymodbus4j.func.response.ErrorFunctionResponse;
import com.github.zengfr.easymodbus4j.func.response.ReadCoilsResponse;
import com.github.zengfr.easymodbus4j.func.response.ReadDiscreteInputsResponse;
import com.github.zengfr.easymodbus4j.func.response.ReadHoldingRegistersResponse;
import com.github.zengfr.easymodbus4j.func.response.ReadInputRegistersResponse;
import com.github.zengfr.easymodbus4j.func.response.WriteMultipleCoilsResponse;
import com.github.zengfr.easymodbus4j.func.response.WriteMultipleRegistersResponse;
import com.github.zengfr.easymodbus4j.func.response.WriteSingleCoilResponse;
import com.github.zengfr.easymodbus4j.func.response.WriteSingleRegisterResponse;
import com.github.zengfr.easymodbus4j.protocol.ModbusFunction;
import com.github.zengfr.easymodbus4j.protocol.ModbusFunctionCode;
import com.github.zengfr.easymodbus4j.protocol.tcp.ModbusFrame;
import com.github.zengfr.easymodbus4j.protocol.tcp.ModbusHeader;
import com.github.zengfr.easymodbus4j.util.ModbusFunctionUtil;


/**
 * @author zengfr QQ:362505707/1163551688 Email:zengfr3000@qq.com
 *         https://github.com/zengfr/easymodbus4j
 */
public class ModbusTcpDecoder extends ByteToMessageDecoder {
	private static final InternalLogger logger = InternalLoggerFactory.getInstance(ModbusTcpDecoder.class);
	private final boolean isSlave;

	public ModbusTcpDecoder(boolean isSlave) {
		this.isSlave = isSlave;
	}

	@Override
	protected void decode(ChannelHandlerContext ctx, ByteBuf buffer, List<Object> out) {
		logger.debug("decode");
		if (buffer.capacity() < ModbusConsts.MBAP_LENGTH + 1 /* Function Code */) {
			return;
		}
		buffer.markReaderIndex();
		ModbusFrame frame = decodeFrame(buffer, this.isSlave);
		if (frame != null) {
			out.add(frame);
		} else {
			buffer.resetReaderIndex();
			ctx.fireChannelRead(buffer);
		}
	}

	public static ModbusFrame decodeFrame(ByteBuf buffer, boolean isSlave) {
		if (buffer.capacity() < ModbusConsts.MBAP_LENGTH + 1) {
			return null;
		}

		ModbusHeader mbapHeader = ModbusHeader.decode(buffer);
		ModbusFunction function = decodeFunction(buffer, isSlave);
		ModbusFrame frame = new ModbusFrame(mbapHeader, function);
		return frame;
	}

	public static ModbusFunction decodeFunction(ByteBuf buffer, boolean isRequest) {
		ModbusFunction function = null;
		short functionCode = buffer.readUnsignedByte();
		if (isRequest) {
			function = decodeReqFunction(functionCode);
		} else {
			function = decodeRespFunction(functionCode);
		}
		if (ModbusFunctionUtil.isError(functionCode)) {
			function = new ErrorFunctionResponse(functionCode);
		} else if (function == null) {
			function = new ErrorFunctionResponse(functionCode, (short) 1);
		}
		function.decode(buffer.readBytes(buffer.readableBytes()));
		return function;
	}

	protected static ModbusFunction decodeReqFunction(short functionCode) {
		ModbusFunction function = null;
		switch (functionCode) {
		case ModbusFunctionCode.READ_COILS:
			function = new ReadCoilsRequest();
			break;
		case ModbusFunctionCode.READ_DISCRETE_INPUTS:
			function = new ReadDiscreteInputsRequest();
			break;
		case ModbusFunctionCode.READ_INPUT_REGISTERS:
			function = new ReadInputRegistersRequest();
			break;
		case ModbusFunctionCode.READ_HOLDING_REGISTERS:
			function = new ReadHoldingRegistersRequest();
			break;
		case ModbusFunctionCode.WRITE_SINGLE_COIL:
			function = new WriteSingleCoilRequest();
			break;
		case ModbusFunctionCode.WRITE_SINGLE_REGISTER:
			function = new WriteSingleRegisterRequest();
			break;
		case ModbusFunctionCode.WRITE_MULTIPLE_COILS:
			function = new WriteMultipleCoilsRequest();
			break;
		case ModbusFunctionCode.WRITE_MULTIPLE_REGISTERS:
			function = new WriteMultipleRegistersRequest();
			break;
		}

		return function;
	}

	public static ModbusFunction decodeRespFunction(short functionCode) {
		ModbusFunction function = null;
		switch (functionCode) {
		case ModbusFunctionCode.READ_COILS:
			function = new ReadCoilsResponse();
			break;
		case ModbusFunctionCode.READ_DISCRETE_INPUTS:
			function = new ReadDiscreteInputsResponse();
			break;
		case ModbusFunctionCode.READ_INPUT_REGISTERS:
			function = new ReadInputRegistersResponse();
			break;
		case ModbusFunctionCode.READ_HOLDING_REGISTERS:
			function = new ReadHoldingRegistersResponse();
			break;
		case ModbusFunctionCode.WRITE_SINGLE_COIL:
			function = new WriteSingleCoilResponse();
			break;
		case ModbusFunctionCode.WRITE_SINGLE_REGISTER:
			function = new WriteSingleRegisterResponse();
			break;
		case ModbusFunctionCode.WRITE_MULTIPLE_COILS:
			function = new WriteMultipleCoilsResponse();
			break;
		case ModbusFunctionCode.WRITE_MULTIPLE_REGISTERS:
			function = new WriteMultipleRegistersResponse();
			break;
		}

		return function;
	}
}
