/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.github.zengfr.easymodbus4j.codec.rtu;

import java.util.List;

import com.github.zengfr.easymodbus4j.codec.ModbusPduCodec;
import com.github.zengfr.easymodbus4j.common.util.RtuCrcUtil;
import com.github.zengfr.easymodbus4j.ModbusConsts;
import com.github.zengfr.easymodbus4j.protocol.ModbusFunction;
import com.github.zengfr.easymodbus4j.protocol.tcp.ModbusFrame;
import com.github.zengfr.easymodbus4j.protocol.tcp.ModbusHeader;
import com.github.zengfr.easymodbus4j.util.ModbusTransactionIdUtil;

import io.netty.buffer.ByteBuf;
import io.netty.channel.ChannelHandlerContext;
import io.netty.handler.codec.ByteToMessageCodec;
import io.netty.util.internal.logging.InternalLogger;
import io.netty.util.internal.logging.InternalLoggerFactory;

/**
 * @author zengfr QQ:362505707/1163551688 Email:zengfr3000@qq.com
 *         https://github.com/zengfr/easymodbus4j
 */
public class ModbusRtuCodec2 extends   ByteToMessageCodec<ModbusFrame> {
	private static final InternalLogger logger = InternalLoggerFactory.getInstance(ModbusRtuCodec2.class);
	protected final ModbusPduCodec pduCodec;
	public ModbusRtuCodec2(ModbusPduCodec pduCodec) {
		this.pduCodec = pduCodec;
	}

	@Override
	protected void encode(ChannelHandlerContext ctx, ModbusFrame msg, ByteBuf out) throws Exception {
		logger.debug("encode");
		ByteBuf sendBuf = ctx.alloc().heapBuffer(1 + (msg.getHeader().getLength() - 1) + 2);
		sendBuf.writeByte(msg.getHeader().getUnitIdentifier());
		sendBuf.writeBytes(msg.getFunction().encode());
		writeRtuCRC(sendBuf);
		//out.writeBytes(sendBuf);
		ctx.writeAndFlush(sendBuf);
	}

	private void writeRtuCRC(ByteBuf buffer) {
		int startReaderIndex = buffer.readerIndex();
		int crc = RtuCrcUtil.calculateCRC(buffer);
		buffer.readerIndex(startReaderIndex);
		buffer.writeByte((byte) (0xff & (crc >> 8)));
		buffer.writeByte((byte) (0xff & crc));
	}

	@Override
	protected void decode(ChannelHandlerContext ctx, ByteBuf in, List<Object> out) throws Exception {
		logger.debug("decode");
		int startIndex = in.readerIndex();
		while (in.readableBytes() >= ModbusRtuCodecUtil.getMessageMinSize() && in.readableBytes() >= ModbusRtuCodecUtil.getMessageLength(in, startIndex)) {
			short unitId = in.readUnsignedByte();
			ModbusFunction function = this.pduCodec.decode(in);
			int crc = in.readUnsignedShort();

			int transactionId = -ModbusTransactionIdUtil.calculateTransactionId();//rtu must be <0
			int pduLength = function.calculateLength();
			int protocolIdentifier = ModbusConsts.DEFAULT_PROTOCOL_IDENTIFIER;

			ModbusHeader mbapHeader = new ModbusHeader(transactionId, protocolIdentifier, pduLength, unitId);

			ModbusFrame frame = new ModbusFrame(mbapHeader, function);
			if (frame != null) {
				out.add(frame);
			}
			startIndex = in.readerIndex();
		}
	}

	 
}
