/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.github.zengfr.easymodbus4j.codec.ping;

import java.nio.charset.Charset;
import java.util.List;

import io.netty.buffer.ByteBuf;
import io.netty.channel.ChannelHandlerContext;
import io.netty.handler.codec.MessageToMessageDecoder;
import io.netty.util.internal.logging.InternalLogger;
import io.netty.util.internal.logging.InternalLoggerFactory;

/**
 * @author zengfr QQ:362505707/1163551688 Email:zengfr3000@qq.com
 *         https://github.com/zengfr/easymodbus4j
 */
public class ModbusPingDecoder extends MessageToMessageDecoder<ByteBuf> {
	private static final InternalLogger logger = InternalLoggerFactory.getInstance(ModbusPingDecoder.class);
	private final Charset charset;
	private final String heartBeat;

	public ModbusPingDecoder(String heartBeat, Charset charset) {
		if (charset == null || heartBeat == null) {
			throw new NullPointerException("heartBeat/charset");
		}
		this.heartBeat = heartBeat;
		this.charset = charset;
	}

	@Override
	protected void decode(ChannelHandlerContext ctx, ByteBuf in, List<Object> out) throws Exception {
		in.retain();
		boolean success = false;
		int heartBeatByteLen = heartBeat.length() * 2;
		if (in.readableBytes() >= heartBeatByteLen) {
			
			byte[] headArray = new byte[heartBeatByteLen];
			in.markReaderIndex();
			in.readBytes(headArray);
			String head = new String(headArray, charset);
			success = heartBeat.equalsIgnoreCase(head);
			if (success) {
				out.add(head);
			} else {
				in.resetReaderIndex();
			}
		}
		logger.debug(String.format("decode isHeartBeat:%s", success));
		if(!success) {
			ctx.fireChannelRead(in);
		}
	}
}
