package com.ifourthwall.job.redis.scheduler.config;

import com.fasterxml.jackson.annotation.JsonAutoDetect;
import com.fasterxml.jackson.annotation.PropertyAccessor;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.github.davidmarquis.redisscheduler.RedisTaskScheduler;
import com.github.davidmarquis.redisscheduler.drivers.spring.RedisTemplateDriver;
import com.ifourthwall.job.redis.scheduler.IFWJob;
import com.ifourthwall.job.redis.scheduler.IFWTaskTriggerListener;
import io.lettuce.core.ClientOptions;
import io.lettuce.core.RedisClient;
import io.lettuce.core.RedisURI;
import io.lettuce.core.resource.ClientResources;
import io.lettuce.core.resource.DefaultClientResources;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.boot.autoconfigure.condition.ConditionalOnProperty;
import org.springframework.boot.context.properties.EnableConfigurationProperties;
import org.springframework.context.ApplicationListener;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.context.event.ContextClosedEvent;
import org.springframework.data.redis.connection.RedisPassword;
import org.springframework.data.redis.connection.RedisStandaloneConfiguration;
import org.springframework.data.redis.connection.jedis.JedisConnectionFactory;
import org.springframework.data.redis.connection.lettuce.LettuceClientConfiguration;
import org.springframework.data.redis.connection.lettuce.LettuceConnectionFactory;
import org.springframework.data.redis.connection.lettuce.LettucePoolingClientConfiguration;
import org.springframework.data.redis.core.RedisTemplate;
import org.springframework.data.redis.serializer.Jackson2JsonRedisSerializer;
import org.springframework.data.redis.serializer.StringRedisSerializer;
import org.springframework.util.CollectionUtils;

import javax.annotation.PostConstruct;
import java.util.ArrayList;
import java.util.List;

/**
 * @Author: hao.dai
 * @Date: 2020/1/11 14:11
 * @Description:
 */
@Configuration
@ConditionalOnProperty(prefix = "ifw.job.config", value = "enable", havingValue = "true", matchIfMissing = true)
@EnableConfigurationProperties(value = {RedisSchedulerJobProperty.class})
@Slf4j
public class IFWRedisSchedulerAutoConfiguration implements ApplicationListener<ContextClosedEvent> {
    @Autowired
    private RedisSchedulerJobProperty redisSchedulerJobProperty;

    @Autowired
    private List<IFWJob> ifwJobs;

    private List<RedisTaskScheduler> redisTaskSchedulers = new ArrayList<>();

    @PostConstruct
    public void init() {
        if (CollectionUtils.isEmpty(ifwJobs)) {
            log.warn("There is no job need to start.");
            return;
        }

        RedisTemplate redisTemplate = createRedisTemplate();
        RedisTaskScheduler scheduler = new RedisTaskScheduler(new RedisTemplateDriver(redisTemplate),
                new IFWTaskTriggerListener(ifwJobs));
        scheduler.setMaxRetriesOnConnectionFailure(redisSchedulerJobProperty.getMaxRetriesOnConnectionFailure());
        scheduler.setSchedulerName(redisSchedulerJobProperty.getApplicationName());
        scheduler.setPollingDelayMillis(redisSchedulerJobProperty.getPollingDelayMillis());
        scheduler.start();
        log.info("Succeed to Start job client.");
        redisTaskSchedulers.add(scheduler);
    }

    @Bean("jobRedisTemplate")
    public RedisTemplate createRedisTemplate() {
        JedisConnectionFactory redisConnectionFactory = new JedisConnectionFactory();
        redisConnectionFactory.setHostName(redisSchedulerJobProperty.getHost());
        redisConnectionFactory.setPort(redisSchedulerJobProperty.getPort());
        redisConnectionFactory.setDatabase(redisSchedulerJobProperty.getDatabase());
        redisConnectionFactory.setPassword(redisSchedulerJobProperty.getPassword());

        RedisTemplate<Object, Object> template = new RedisTemplate<>();
        Jackson2JsonRedisSerializer<Object> jackson2JsonRedisSerializer = new
                Jackson2JsonRedisSerializer<Object>(Object.class);
        ObjectMapper om = new ObjectMapper();

        om.setVisibility(PropertyAccessor.ALL, JsonAutoDetect.Visibility.ANY);
        om.enableDefaultTyping(ObjectMapper.DefaultTyping.NON_FINAL);
        jackson2JsonRedisSerializer.setObjectMapper(om);
        template.setKeySerializer(new StringRedisSerializer());
        template.setHashKeySerializer(new StringRedisSerializer());
        template.setValueSerializer(jackson2JsonRedisSerializer);
        template.setHashValueSerializer(jackson2JsonRedisSerializer);
        template.setConnectionFactory(redisConnectionFactory);
        return template;
    }


    @Override
    public void onApplicationEvent(ContextClosedEvent event) {
        if (!CollectionUtils.isEmpty(redisTaskSchedulers)) {
            redisTaskSchedulers.stream().forEach(redisTaskScheduler -> {
                redisTaskScheduler.stop();
            });
        }
    }
}
