package com.ifourthwall.dbm.sentry.service.impl;

import com.ifourthwall.common.PlatformCodeEnum;
import com.ifourthwall.common.base.BaseResponse;
import com.ifourthwall.common.base.IFWPageInfo;
import com.ifourthwall.common.base.IFWUser;
import com.ifourthwall.common.cache.RedisKeyPrefixUtil;
import com.ifourthwall.common.exception.BizException;
import com.ifourthwall.common.kafka.TopicConstants;
import com.ifourthwall.common.utils.*;
import com.ifourthwall.dbm.asset.dto.*;
import com.ifourthwall.dbm.asset.dto.app.UpdateAppIdQuDTO;
import com.ifourthwall.dbm.asset.dto.metric.queryDataPointIdDTO;
import com.ifourthwall.dbm.asset.dto.metric.queryDataPointIdQuDTO;
import com.ifourthwall.dbm.asset.facade.MonitorFacade;
import com.ifourthwall.dbm.common.util.MathUtils;
import com.ifourthwall.dbm.sentry.bo.*;
import com.ifourthwall.dbm.sentry.bo.alarm.*;
import com.ifourthwall.dbm.sentry.bo.app.AppReqBO;
import com.ifourthwall.dbm.sentry.bo.app.AppRespBO;
import com.ifourthwall.dbm.sentry.bo.metric.MetricValueRecordListBO;
import com.ifourthwall.dbm.sentry.bo.metric.MetricValueRecordListReqBO;
import com.ifourthwall.dbm.sentry.domain.*;
import com.ifourthwall.dbm.sentry.service.MonitorService;
import com.ifourthwall.dbm.project.dto.GetSonSpaceQueryDTO;
import com.ifourthwall.dbm.sentry.util.DataUtils;
import com.ifourthwall.dbm.user.dto.ServiceWetherPurchaseReqDTO;
import com.ifourthwall.kafka.IFWKafkaClient;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.apache.dubbo.config.annotation.Reference;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.data.redis.core.RedisTemplate;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import javax.annotation.Resource;
import java.math.BigDecimal;
import java.math.RoundingMode;

import java.text.SimpleDateFormat;
import java.util.*;
import java.util.stream.Collectors;

/**
 * @Author: YangPeng
 * @Date: 2019/12/11 10:29
 * @Description:
 */
@Slf4j
@Service("MonitorServiceImpl")
public class MonitorServiceImpl implements MonitorService {

    /**
     * 启用告警
     */
    public static final String ENABLE_ALARM = "1";
    /**
     * 关闭告警
     */
    public static final String DISABLE_ALARM = "0";

    @Autowired
    private RedisTemplate redisTemplate;

    @Reference(version = "1.0.0")
    private MonitorFacade monitorFacade;

    @Resource(name = "MonitorRepository")
    private MonitorRepository monitorRepository;

    @Resource(name = "ProjectRepository")
    private ProjectRepository projectRepository;

    @Resource(name = "AssetRepository")
    private AssetRepository assetRepository;

    @Resource(name = "SpaceRepository")
    private SpaceRepository spaceRepository;

    @Resource(name = "TaskRepository")
    private TaskRepository taskRepository;

    @Resource(name = "UserRepository")
    private UserRepository userRepository;

    @Value("${robot.system.id}")
    private String robotId;

    @Value("${callback.url}")
    private String callbackUrl;

    @Value("${callback.token}")
    private String callbackToken;

    @Value("${robot.system.position.id}")
    private String robotPositionId;

    @Value("${ifw.sentry.service.id}")
    private String sentryServiceId;


    @Autowired
    private GodzillaServerRepository godzillaServerRepository;


    //创建工单
    @Resource(name = "IFWKafkaClient-" + TopicConstants.dbmSentryCreateTask)
    private IFWKafkaClient ifwKafkaClientCreateTask;

    //当有自动告警发生时，把该空间的告警数+1 或-1
    @Resource(name = "IFWKafkaClient-" + TopicConstants.DBM_SENTRY_UPDATE_SPACE_SPECIAL_ALARM_NUM)
    private IFWKafkaClient ifwKafkaClientUpdateAlarmNumber;


    /**
     * 新增修改监控
     *
     * @param upInsertMonitor
     * @param user
     * @return com.ifourthwall.common.base.BaseResponse
     * @throws
     * @Author: YangPeng
     * @Date: 2019/12/11 10:27
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public BaseResponse upInsertMonitor(UpInsertMonitorBasisDTO upInsertMonitor, IFWUser user) {
        BaseResponse response = new BaseResponse<>();
        try {
            //TODO 优化：需要增加查询是否存在监控对象
            //新增sentry基础信息
            String tenantId = CheckAccessUtils.judgeTenantId(upInsertMonitor.getTenantId(), user);
            upInsertMonitor.setTenantId(tenantId);
            UpInsertMonitorDTO upInsert = new UpInsertMonitorDTO();
            BeanUtils.copyProperties(upInsertMonitor, upInsert);
            upInsert.setTenantId(tenantId);
            if (upInsert.getType() == 1) {
                upInsert.setCreateBy(user.getUserId());
            }
            if (upInsert.getType() == 2) {
                upInsert.setUpdateBy(user.getUserId());
            }
            UpInsertMonitorDoBO query = IFWBeanCopyUtil.map(upInsert, UpInsertMonitorDoBO.class);
            monitorRepository.upInsertMonitor(query);
            response.setRetCode(PlatformCodeEnum.EXECUTE_SUCCESS.getCode());
        } catch (BizException bizException) {
            response.setRetCode(bizException.getCode());
            response.setRetMsg(bizException.getMessage());
            response.setFlag(false);
        } catch (Throwable e) {
            log.error("系统异常", e);
            response.setRetCode(PlatformCodeEnum.SYS_ERROR.getCode());
            response.setRetMsg(PlatformCodeEnum.SYS_ERROR.getDesc());
            response.setFlag(false);
        }
        return response;
    }


    /**
     * 更新哥斯拉信息
     *
     * @param upInsertMonitor
     * @param user
     * @return com.ifourthwall.common.base.BaseResponse
     * @throws
     * @Author: YangPeng
     * @Date: 2020/1/16 17:08
     */
    @Override
    @Transactional
    public BaseResponse upInsertIndex(UpInsertMonitorBasisDTO upInsertMonitor, IFWUser user) {
        BaseResponse response = new BaseResponse<>();
        try {
            String tenantId = CheckAccessUtils.judgeTenantId(upInsertMonitor.getTenantId(), user);
            upInsertMonitor.setTenantId(tenantId);
            if (DataUtils.isListAvali(upInsertMonitor.getMonitorMetricInfo())) {
                //查询项目名称
                GetProjectSpaceInfoQueryBO queryProjectName = new GetProjectSpaceInfoQueryBO();
                queryProjectName.setTenantId(upInsertMonitor.getTenantId());
                queryProjectName.setProjectId(upInsertMonitor.getProjectId());
                queryProjectName.setLanguageCode(upInsertMonitor.getLanguageCode());
                QueryProjectPagePO projectSpaceInfo = projectRepository.getProjectSpaceInfo(queryProjectName);
                if (projectSpaceInfo != null) {
                    //新增哥斯拉数据
                    UpInGodzillaMonitor upInGodzillaMonitor = new UpInGodzillaMonitor();
                    ArrayList<GodzillaDatapointInfo> godzillaDataPoint = new ArrayList<>();
                    //查询资产信息
                    AssetInfoQueryDoBO assetInfoQueryDoBO = new AssetInfoQueryDoBO();
                    assetInfoQueryDoBO.setLanguageCode(upInsertMonitor.getLanguageCode());
                    assetInfoQueryDoBO.setAssetId(upInsertMonitor.getAssetId());
                    assetInfoQueryDoBO.setProjectId(upInsertMonitor.getProjectId());
                    QueryAssetInfoDoBO queryAssetInfoDoBO = assetRepository.queryAssetInfo(assetInfoQueryDoBO);
                    if (queryAssetInfoDoBO != null) {
                        //查询空间信息
                        GetNewPathBatchQueryBO queryName = new GetNewPathBatchQueryBO();
                        ArrayList<String> ids = new ArrayList<>();
                        ids.add(queryAssetInfoDoBO.getProjectSpaceId());
                        queryName.setProjectSpaceId(ids);
                        queryName.setProjectId(upInsertMonitor.getProjectId());
                        queryName.setLanguageCode(upInsertMonitor.getLanguageCode());
                        queryName.setTenantId(tenantId);
                        List<ProjectSpacePathBO> newPathBatch = spaceRepository.getNewPathBatch(queryName);
                        if (DataUtils.isListAvali(newPathBatch)) {
                            ArrayList<String> ruleIds = new ArrayList<>();
                            ruleIds.add(upInsertMonitor.getAlarmRuleId());
                            //默认绑定id
                            QueryAppIdQueryDoBO queryAppIdQueryDoBO = new QueryAppIdQueryDoBO();
                            queryAppIdQueryDoBO.setTenantId(tenantId);
                            QueryAppIdBO queryAppIdBO = monitorRepository.queryAppId(queryAppIdQueryDoBO);
                            if (queryAppIdBO != null && queryAppIdBO.getDefaultAlarmRuleId() == null) {
                                AlarmRuleInsertBO insert = new AlarmRuleInsertBO();
                                insert.setAppId(queryAppIdBO.getAppId());
                                insert.setCallbackUrl(callbackUrl);
                                insert.setCallbackToken(callbackToken);
                                insert.setAlarmChannel("callback");
                                insert.setAlarmRuleName("sentry默认告警模板-" + user.getOrganizationName());
                                insert.setAlarmCycle("0");
                                log.info("addAlarmRule入参：" + insert);
                                String ruleId = godzillaServerRepository.addAlarmRule(insert);
                                UpdateAppIdQuDTO updateAppIdQuDTO = new UpdateAppIdQuDTO();
                                updateAppIdQuDTO.setTenantId(tenantId);
                                updateAppIdQuDTO.setRuleId(ruleId);
                                monitorRepository.updateAppId(updateAppIdQuDTO);
                                ruleIds.add(ruleId);
                            }else {
                                ruleIds.add(queryAppIdBO.getDefaultAlarmRuleId());
                            }
                            for (MonitorMetricBasisDTO monitorMetric : upInsertMonitor.getMonitorMetricInfo()) {
                                DatapointAndAlarmRuleBindReqBO datapointAndAlarmRuleBindReqBO = new DatapointAndAlarmRuleBindReqBO();
                                //查询monitor_metric_id
                                QueryDataByMonitorQuDoBO queryDataByMonitorQuDoBO = new QueryDataByMonitorQuDoBO();
                                queryDataByMonitorQuDoBO.setDataPointId(monitorMetric.getDataPointId());
                                queryDataByMonitorQuDoBO.setTenantId(tenantId);
                                QueryDataByMonitorDoBO queryDataByMonitorDoBO = monitorRepository.queryDataByMonitor(queryDataByMonitorQuDoBO);
                                if (queryDataByMonitorDoBO.getMonitorId() != null) {

                                    GodzillaDatapointInfo godzillaInfo = new GodzillaDatapointInfo();
                                    godzillaInfo.setDataPointId(monitorMetric.getDataPointId());
                                    StringBuilder godzillaName = new StringBuilder();
                                    //告警模板分三国语言
                                    if ("CN".equals(upInsertMonitor.getLanguageCode())) {
                                        godzillaName.append(upInsertMonitor.getAssetName() + ",监测指标:" + monitorMetric.getMonitorMetricName() + ",项目名称:" + projectSpaceInfo.getProjectName() + ",空间位置：" + newPathBatch.get(0).getNameStr());
                                    }
                                    if ("JP".equals(upInsertMonitor.getLanguageCode())) {
                                        godzillaName.append(upInsertMonitor.getAssetName() + ",モニタリング指標:" + monitorMetric.getMonitorMetricName() + ",プロジェクト名:" + projectSpaceInfo.getProjectName() + ",空間位置：" + newPathBatch.get(0).getNameStr());
                                    }
                                    if ("EN".equals(upInsertMonitor.getLanguageCode())) {
                                        godzillaName.append(upInsertMonitor.getAssetName() + ",Monitor Metric Name:" + monitorMetric.getMonitorMetricName() + ",Project Name:" + projectSpaceInfo.getProjectName() + ",Space Name：" + newPathBatch.get(0).getNameStr());
                                    }
                                    godzillaInfo.setDataPointName(godzillaName.toString());
                                    godzillaInfo.setUnit(monitorMetric.getUnit());

                                    if (upInsertMonitor.getAutoAlert() == 1) {
                                        datapointAndAlarmRuleBindReqBO.setAliasEnableAlarm(ENABLE_ALARM);
                                    }
                                    if (upInsertMonitor.getAutoAlert() == 0) {
                                        datapointAndAlarmRuleBindReqBO.setAliasEnableAlarm(DISABLE_ALARM);
                                    }
                                    godzillaDataPoint.add(godzillaInfo);
                                    //哥斯拉接口暂不支持批量绑定告警规则
                                    datapointAndAlarmRuleBindReqBO.setAlarmRuleIds(String.join(",",ruleIds));
                                    datapointAndAlarmRuleBindReqBO.setDataPointId(monitorMetric.getDataPointId());
                                    godzillaServerRepository.bindDatapointAndAlarmRule(datapointAndAlarmRuleBindReqBO);
                                    //更新最大值最小值
                                    if (DataUtils.isListAvali(monitorMetric.getAlarmInfo())) {
                                        for (AlertRulesDTO alertRulesDTO : monitorMetric.getAlarmInfo()) {
                                            AddMaxAndMinBO updateMaxAndMinBO = new AddMaxAndMinBO();
                                            updateMaxAndMinBO.setAlarmMaximumValue(alertRulesDTO.getAlarmMaximumValue());
                                            updateMaxAndMinBO.setAlarmMinimumValue(alertRulesDTO.getAlarmMinimumValue());
                                            updateMaxAndMinBO.setDataPointId(monitorMetric.getDataPointId());
                                            updateMaxAndMinBO.setCompareTarget(monitorMetric.getDataPointId());
                                            updateMaxAndMinBO.setCompareDuration(alertRulesDTO.getCompareDuration() * 60 * 1000);
                                            godzillaServerRepository.addMaxAndMin(updateMaxAndMinBO);
                                        }
                                    }
                                }
                            }
                        }
                    } else {
                        throw new BizException(PlatformCodeEnum.ASSET_IS_NULL.getDesc(), PlatformCodeEnum.ASSET_IS_NULL.getCode());
                    }
                    upInGodzillaMonitor.setDatapointList(godzillaDataPoint);
                    godzillaServerRepository.updateDataPointName(upInGodzillaMonitor);
                }
            }
            response.setRetCode(PlatformCodeEnum.EXECUTE_SUCCESS.getCode());
        } catch (BizException bizException) {
            response.setRetCode(PlatformCodeEnum.CREATE_FAILED.getCode());
            response.setRetMsg(PlatformCodeEnum.CREATE_FAILED.getDesc());
            response.setFlag(false);
        } catch (Throwable e) {
            log.error("系统异常", e);
            response.setRetCode(PlatformCodeEnum.SYS_ERROR.getCode());
            response.setRetMsg(PlatformCodeEnum.SYS_ERROR.getDesc());
            response.setFlag(false);
        }
        return response;
    }

    /**
     * 查询监控详情
     *
     * @param monitorInfo
     * @return com.ifourthwall.common.base.BaseResponse<com.ifourthwall.dbm.sentry.dto.QueryMonitorInfoDTO>
     * @throws
     * @Author: YangPeng
     * @Date: 2019/12/12 11:03
     */
    @Override
    public BaseResponse<QueryMonitorInfoDTO> queryMonitorInfo(QueryMonitorInfoQueryBasisDTO monitorInfo, IFWUser user) {
        BaseResponse<QueryMonitorInfoDTO> response = new BaseResponse<>();
        try {
            String tenantId = CheckAccessUtils.judgeTenantId(monitorInfo.getTenantId(), user);
            monitorInfo.setTenantId(tenantId);
            QueryAppIdQueryDoBO queryAppId = new QueryAppIdQueryDoBO();
            queryAppId.setTenantId(tenantId);
            QueryAppIdBO queryAppIdBO = monitorRepository.queryAppId(queryAppId);
            //查询监控对象信息
            QueryMonitorInfoQueryBasisDoBO query = new QueryMonitorInfoQueryBasisDoBO();
            query.setAssetId(monitorInfo.getAssetId());
            query.setTenantId(tenantId);
            query.setProjectId(monitorInfo.getProjectId());
            query.setLanguageCode(monitorInfo.getLanguageCode());
            QueryMonitorInfoDoBO queryMonitorInfoDoBOPageDTO = monitorRepository.queryMonitorInfo(query);
            if (queryMonitorInfoDoBOPageDTO == null) {
                return response;
            }
            QueryMonitorInfoDTO result = IFWBeanCopyUtil.map(queryMonitorInfoDoBOPageDTO, QueryMonitorInfoDTO.class);
            //查询空间名称
            GetNewPathBatchQueryBO querySpaceName = new GetNewPathBatchQueryBO();
            ArrayList<String> spaceId = new ArrayList<>();
            spaceId.add(queryMonitorInfoDoBOPageDTO.getSpaceId());
            querySpaceName.setProjectId(monitorInfo.getProjectId());
            querySpaceName.setProjectSpaceId(spaceId);
            querySpaceName.setLanguageCode(monitorInfo.getLanguageCode());
            List<ProjectSpacePathBO> newPathBatch = spaceRepository.getNewPathBatch(querySpaceName);
            if (!DataUtils.isListAvali(newPathBatch)) {
                result.setSpaceName("");
            } else {
                result.setSpaceName(newPathBatch.get(0).getNameStr());
            }
            if (DataUtils.isListAvali(result.getMonitorMetricInfo())) {
                for (MonitorInfoDTO monitorInfoDTO : result.getMonitorMetricInfo()) {
                    if (!StringUtils.isEmpty(monitorInfoDTO.getDataPointId())) {
                        //查询哥斯拉相关数据
                        QueryRuleAndStartBO godzillaResult = new QueryRuleAndStartBO();
                        godzillaResult.setDataPointId(monitorInfoDTO.getDataPointId());
                        QueryRuleAndStartDoBO godzillaInfo = godzillaServerRepository.queryRuleAndStart(godzillaResult);
                        if (godzillaInfo != null) {
                            if (!StringUtils.isEmpty(godzillaInfo.getAlarmRuleId())) {
                                String[] split = godzillaInfo.getAlarmRuleId().split(",");
                                for (String ruleId : split) {
                                    if (!ruleId.equals(queryAppIdBO.getDefaultAlarmRuleId())) {
                                        result.setAlarmRuleId(ruleId);
                                    }
                                }
                            }
                            monitorInfoDTO.setUnit(godzillaInfo.getUnit());
//                            monitorInfoDTO.setAlarmInfo(godzillaInfo.getAlarmInfo());
                            result.setAlarmRuleName(godzillaInfo.getAlarmRuleName());
                            result.setAliasEnableAlarm(godzillaInfo.getAliasEnableAlarm());
                        }
                    }
                }
            }
            response.setData(result);
            response.setRetCode(PlatformCodeEnum.EXECUTE_SUCCESS.getCode());
        } catch (BizException bizException) {
            response.setRetCode(bizException.getCode());
            response.setRetMsg(bizException.getMessage());
            response.setFlag(false);
        } catch (Throwable e) {
            log.error("系统异常", e);
            response.setRetCode(PlatformCodeEnum.SYS_ERROR.getCode());
            response.setRetMsg(PlatformCodeEnum.SYS_ERROR.getDesc());
            response.setFlag(false);
        }
        return response;
    }

    /**
     * 查询监控列表
     *
     * @param monitorList
     * @param user
     * @return com.ifourthwall.common.base.BaseResponse<com.ifourthwall.dbm.asset.dto.PageDTO < com.ifourthwall.dbm.asset.dto.QueryMonitorListDTO>>
     * @throws
     * @Author: YangPeng
     * @Date: 2020/3/17 10:21
     */
    @Override
    public BaseResponse<PageDTO<QueryMonitorListDTO>> queryMonitorList(QueryMonitorListQueryDTO monitorList, IFWUser user) {
        BaseResponse<PageDTO<QueryMonitorListDTO>> response = new BaseResponse<>();
        PageDTO<QueryMonitorListDTO> resultPage = new PageDTO<>();
        try {
            String tenantId = CheckAccessUtils.judgeTenantId(monitorList.getTenantId(), user);
            monitorList.setTenantId(tenantId);
            QueryMonitorListDoQueryBO queryMonitorList = new QueryMonitorListDoQueryBO();
            //根据空间名称搜索
            if (!StringUtils.isEmpty(monitorList.getSpaceName())) {
                ProjectSpaceSubListQueryDoBO queryName = new ProjectSpaceSubListQueryDoBO();
                queryName.setProjectId(monitorList.getProjectId());
                queryName.setNameLike(monitorList.getSpaceName());
                queryName.setLanguageCode(monitorList.getLanguageCode());
                queryName.setTenantId(monitorList.getTenantId());
                List<ProjectSpaceBaseDoBO> sonSpaceList = spaceRepository.getSonSpaceList(queryName);
                if (sonSpaceList != null) {
                    List<String> ids = sonSpaceList.stream().map(ProjectSpaceBaseDoBO::getProjectSpaceId).collect(Collectors.toList());
                    queryMonitorList.setSpaceId(ids);
                }
            }
            //是否根据空间查询
            if ("1".equals(monitorList.getIsOrNot())) {
                if (monitorList.getSpaceId() != null) {
                    GetSonSpaceQueryDTO querySpace = new GetSonSpaceQueryDTO();
                    querySpace.setSpaceId(monitorList.getSpaceId());
                    GetSonSpaceBO allSpace = spaceRepository.getSonpace(IFWBeanCopyUtil.map(querySpace, GetSonSpaceQueryBO.class));
                    if (!DataUtils.isListAvali(allSpace.getSpaceId())) {
                        return response;
                    }
                    //放入查询的空间id，后面的查询资产根据空间id来查询
                    allSpace.getSpaceId().add(monitorList.getSpaceId());
                    queryMonitorList.setSpaceId(allSpace.getSpaceId());
                }
            }
            if ("2".equals(monitorList.getIsOrNot())) {
                if (!StringUtils.isEmpty(monitorList.getSpaceId())) {
                    ArrayList<String> spaceIds = new ArrayList<>();
                    spaceIds.add(monitorList.getSpaceId());
                    queryMonitorList.setSpaceId(spaceIds);
                }
            }
            //查询监控对象
            queryMonitorList.setTenantId(tenantId);
            BeanUtils.copyProperties(monitorList, queryMonitorList);
            IFWPageInfo<QueryMonitorListDoBO> monitorPage = monitorRepository.queryMonitorList(queryMonitorList);
            if (monitorPage == null) {
                return response;
            }
            BeanUtils.copyProperties(monitorPage, resultPage, "result");
            List<QueryMonitorListDTO> resultList = IFWBeanCopyUtil.mapAsList(monitorPage.getList(), QueryMonitorListDTO.class);
            //查询空间名字
            GetNewPathBatchQueryBO querySpaceName = new GetNewPathBatchQueryBO();
            ArrayList<String> spaceId = new ArrayList<>();
            for (QueryMonitorListDTO monitor : resultList) {
                spaceId.add(monitor.getProjectSpaceId());
            }
            querySpaceName.setProjectId(monitorList.getProjectId());
            querySpaceName.setProjectSpaceId(spaceId);
            querySpaceName.setLanguageCode(monitorList.getLanguageCode());
            List<ProjectSpacePathBO> newPathBatch = spaceRepository.getNewPathBatch(querySpaceName);
            if (DataUtils.isListAvali(newPathBatch)) {
                for (QueryMonitorListDTO monitor : resultList) {
                    //插入空间名称
                    for (ProjectSpacePathBO spaceName : newPathBatch) {
                        if (monitor.getProjectSpaceId().equals(spaceName.getId())) {
                            monitor.setProjectSpaceName(spaceName.getNameStr());
                        }
                    }
                    //插入哥斯拉信息
                    QueryRuleAndStartBO queryRuleAndStartBO = new QueryRuleAndStartBO();
                    if (DataUtils.isListAvali(monitor.getGodzillaInfo())) {
                        for (GodzillaBindingInfo godzillaBindingInfo : monitor.getGodzillaInfo()) {
                            queryRuleAndStartBO.setDataPointId(godzillaBindingInfo.getDataPointId());
                            QueryRuleAndStartDoBO queryRuleAndStartDoBO = godzillaServerRepository.queryRuleAndStart(queryRuleAndStartBO);
                            if (queryRuleAndStartDoBO != null) {
                                godzillaBindingInfo.setUnit(queryRuleAndStartDoBO.getUnit());
                                godzillaBindingInfo.setCurrentValue(queryRuleAndStartDoBO.getCurrentValue());
                            }
                        }
                    }
                }
            }
            resultPage.setResult(resultList);
            response.setData(resultPage);
            response.setRetCode(PlatformCodeEnum.EXECUTE_SUCCESS.getCode());
        } catch (BizException bizException) {
            response.setRetCode(bizException.getCode());
            response.setRetMsg(bizException.getMessage());
            response.setFlag(false);
        } catch (Throwable e) {
            log.error("系统异常", e);
            response.setRetCode(PlatformCodeEnum.SYS_ERROR.getCode());
            response.setRetMsg(PlatformCodeEnum.SYS_ERROR.getDesc());
            response.setFlag(false);
        }
        return response;
    }

    /**
     * 查询监控翻译配置
     *
     * @param translationDTO
     * @return com.ifourthwall.common.base.BaseResponse<com.ifourthwall.dbm.sentry.dto.QueryMonitorTranslationDTO>
     * @throws
     * @Author: YangPeng
     * @Date: 2019/12/16 16:30
     */
    @Override
    public BaseResponse<QueryMonitorTranslationDTO> queryMonitorTranslation(TranslationDTO translationDTO) {
        return monitorFacade.queryMonitorTranslation(translationDTO);
    }

    /**
     * 查询监控记录列表
     *
     * @param recording
     * @return
     * @throws
     * @Author: YangPeng
     * @Date: 2019/12/19 9:49
     */
    @Override
    public BaseResponse<PageDTO<QueryRecordingListDTO>> queryRecordingList(QueryRecordingQueryBasisDTO recording, IFWUser user) {
        BaseResponse<PageDTO<QueryRecordingListDTO>> response = new BaseResponse<>();
        PageDTO<QueryRecordingListDTO> resultPage = new PageDTO<>();
        try {
            String tenantId = CheckAccessUtils.judgeTenantId(recording.getTenantId(), user);
            recording.setTenantId(tenantId);
            QueryRecordingQueryDoBO queryRecord = new QueryRecordingQueryDoBO();
            //是否根据空间查询
            if (recording.getSpaceId() != null) {
                GetSonSpaceQueryDTO querySpace = new GetSonSpaceQueryDTO();
                querySpace.setSpaceId(recording.getSpaceId());
                GetSonSpaceBO allSpace = spaceRepository.getSonpace(IFWBeanCopyUtil.map(querySpace, GetSonSpaceQueryBO.class));
                if (!DataUtils.isListAvali(allSpace.getSpaceId())) {
                    return response;
                }
                //放入查询的空间id，后面的查询资产根据空间id来查询
                allSpace.getSpaceId().add(recording.getSpaceId());
                queryRecord.setSpaceId(allSpace.getSpaceId());
            }
            //查询监控对象
            queryRecord.setTenantId(tenantId);
            BeanUtils.copyProperties(recording, queryRecord);
            PageDTO<QueryRecordingListDoBO> monitorPage = monitorRepository.queryRecordingList(queryRecord);
            if (!DataUtils.isListAvali(monitorPage.getResult())) {
                return response;
            }
            BeanUtils.copyProperties(monitorPage, resultPage, "result");
            List<QueryRecordingListDTO> resultList = IFWBeanCopyUtil.mapAsList(monitorPage.getResult(), QueryRecordingListDTO.class);
            //查询空间名字
            GetNewPathBatchQueryBO querySpaceName = new GetNewPathBatchQueryBO();
            ArrayList<String> spaceId = new ArrayList<>();
            for (QueryRecordingListDTO monitor : resultList) {
                spaceId.add(monitor.getProjectSpaceId());
            }
            querySpaceName.setProjectId(recording.getProjectId());
            querySpaceName.setProjectSpaceId(spaceId);
            querySpaceName.setLanguageCode(recording.getLanguageCode());
            List<ProjectSpacePathBO> newPathBatch = spaceRepository.getNewPathBatch(querySpaceName);
            if (DataUtils.isListAvali(newPathBatch)) {
                for (QueryRecordingListDTO monitor : resultList) {
                    for (ProjectSpacePathBO spaceName : newPathBatch) {
                        if (monitor.getProjectSpaceId().equals(spaceName.getId())) {
                            monitor.setProjectSpaceName(spaceName.getNameStr());
                        }
                    }
                }
            }
            resultPage.setResult(resultList);
            response.setData(resultPage);
            response.setRetCode(PlatformCodeEnum.EXECUTE_SUCCESS.getCode());
        } catch (BizException bizException) {
            response.setRetCode(bizException.getCode());
            response.setRetMsg(bizException.getMessage());
            response.setFlag(false);
        } catch (Throwable e) {
            log.error("系统异常", e);
            response.setRetCode(PlatformCodeEnum.SYS_ERROR.getCode());
            response.setRetMsg(PlatformCodeEnum.SYS_ERROR.getDesc());
            response.setFlag(false);
        }
        return response;
    }


    /**
     * 接收哥斯拉告警信息（消费）
     *
     * @param
     * @return com.ifourthwall.common.base.BaseResponse
     * @throws
     * @Author: YangPeng
     * @Date: 2019/12/24 10:55
     */
    @Override
//    @Scheduled(cron = "0/15 * * * * ?")
    public BaseResponse generateTaks() {
        BaseResponse response = new BaseResponse<>();
        try {
            Set keys = redisTemplate.opsForHash().keys(RedisKeyPrefixUtil.SENTRY_AUTO_CREATE_WORKSHEET_PREFIX);
            log.info("自动创建创建工单：" + keys);
            if (!keys.isEmpty()) {
                for (Object key : keys) {
                    TaskKafkaBO kafka = (TaskKafkaBO) redisTemplate.opsForHash().get(RedisKeyPrefixUtil.SENTRY_AUTO_CREATE_WORKSHEET_PREFIX, key);
                    if (kafka.getAuto() == 1) {
                        KafkaAlertBO kafkaAlertBO = new KafkaAlertBO();
                        CreateTaskQueryDTO query = new CreateTaskQueryDTO();
                        TaskInsertDoBO taskQuery = new TaskInsertDoBO();
                        ArrayList<TaskSpaceInsertBO> taskSpaceInfo = new ArrayList<>();
                        TaskSpaceInsertBO taskSpace = new TaskSpaceInsertBO();
                        taskSpace.setSpaceId(kafka.getSpaceId());
                        taskSpaceInfo.add(taskSpace);
                        ArrayList<String> operatingId = new ArrayList<>();
                        operatingId.add(robotPositionId);
                        taskQuery.setCreateByPositionId(robotPositionId);
                        taskQuery.setTenantId(kafka.getTenantId());
                        taskQuery.setProjectId(kafka.getProjectId());
                        taskQuery.setTaskTypeId("1");
                        taskQuery.setCreateBy(robotId);
                        taskQuery.setTaskTemplateId(kafka.getTaskTemplateId());
                        taskQuery.setTaskName("sentry自动报警");
                        taskQuery.setTaskDescription("位置:" + kafka.getSpaceName()
                                + "; 报警设备/设施:" + kafka.getAssetName()
                                + "; 报警内容:" + kafka.getMessage() +
                                "; 报警时间:" + kafka.getCollectTime());
                        taskQuery.setCreateByPositionId(robotPositionId);
                        query.setTask(taskQuery);
                        //往kafka扔消息
                        kafkaAlertBO.setAlertId(key.toString());
                        kafkaAlertBO.setTaskBO(query);
                        kafkaAlertBO.getTaskBO().setTaskSpaceList(taskSpaceInfo);
                        String sendInfo = JSONUtils.jacksonToJson(kafkaAlertBO);
                        log.info("自动创建工单，kafka发送数据" + sendInfo);
                        ifwKafkaClientCreateTask.syncProduce(sendInfo);
                    }
                }
                //删除redis数据
                redisTemplate.delete(RedisKeyPrefixUtil.SENTRY_AUTO_CREATE_WORKSHEET_PREFIX);
            }
        } catch (BizException bizException) {
            response.setRetCode(bizException.getCode());
            response.setRetMsg(bizException.getMessage());
            response.setFlag(false);
        } catch (Throwable e) {
            log.error("系统异常", e);
            response.setRetCode(PlatformCodeEnum.SYS_ERROR.getCode());
            response.setRetMsg(PlatformCodeEnum.SYS_ERROR.getDesc());
            response.setFlag(false);
        }
        return response;
    }

    /**
     * 接收哥斯拉告警信息(获取工单参数，更新数据库)
     *
     * @param upMonitorRecordDTO
     * @return com.ifourthwall.common.base.BaseResponse
     * @throws
     * @Author: YangPeng
     * @Date: 2019/12/26 9:47
     */
    @Override
    public BaseResponse updateDataPoint(UpMonitorRecordDTO upMonitorRecordDTO) {
        BaseResponse response = new BaseResponse<>();
        try {
            //从kafka中获取消息进行消费，更新工单状态

        } catch (BizException bizException) {
            response.setRetCode(bizException.getCode());
            response.setRetMsg(bizException.getMessage());
            response.setFlag(false);
        } catch (Throwable e) {
            log.error("系统异常", e);
            response.setRetCode(PlatformCodeEnum.SYS_ERROR.getCode());
            response.setRetMsg(PlatformCodeEnum.SYS_ERROR.getDesc());
            response.setFlag(false);
        }
        return response;
    }


    /**
     * 新增告警通知（透传哥斯拉）
     *
     * @param insetAlertNotice
     * @return com.ifourthwall.common.base.BaseResponse
     * @throws
     * @Author: YangPeng
     * @Date: 2019/12/26 10:20
     */
    @Override
    public BaseResponse insetAlertNotice(InsetAlertNoticeDTO insetAlertNotice, IFWUser user) {
        BaseResponse response = new BaseResponse<>();
        try {
            String tenantId = CheckAccessUtils.judgeTenantId(insetAlertNotice.getTenantId(), user);
            insetAlertNotice.setTenantId(tenantId);
            QueryAppIdQueryDoBO queryAppId = new QueryAppIdQueryDoBO();
            queryAppId.setTenantId(tenantId);
            QueryAppIdBO queryAppIdBO = monitorRepository.queryAppId(queryAppId);
            if (queryAppIdBO == null) {
                throw new BizException("不存在appId，请申请!", "15");
            }
            AlarmRuleInsertBO insert = new AlarmRuleInsertBO();
            ArrayList<String> ruleList = new ArrayList<>();
            if (!StringUtils.isEmpty(insetAlertNotice.getAlarmChannel())) {
                String[] split = insetAlertNotice.getAlarmChannel().split(",");
                for (String rule : split) {
                    if (!"callback".equals(rule)) {
                        ruleList.add(rule);
                    }
                }
            }
            if (DataUtils.isListAvali(ruleList)){
                insert.setAlarmChannel(String.join(",",ruleList));
            }
            insert.setAppId(queryAppIdBO.getAppId());
            insert.setAlarmRuleName(insetAlertNotice.getAlarmRuleName());
            insert.setSmsPhoneNumbers(insetAlertNotice.getSmsPhoneNumbers());
            insert.setEmailToAddresses(insetAlertNotice.getEmailToAddresses());
            insert.setVoiceToPhoneNumber(insetAlertNotice.getVoiceToPhoneNumber());
            insert.setValidTime(insetAlertNotice.getValidTime());
            insert.setAlarmCycle(insetAlertNotice.getAlarmCycle());
            godzillaServerRepository.addAlarmRule(insert);
            response.setRetCode(PlatformCodeEnum.EXECUTE_SUCCESS.getCode());
        } catch (BizException bizException) {
            response.setRetCode(bizException.getCode());
            response.setRetMsg(bizException.getMessage());
            response.setFlag(false);
        } catch (Throwable e) {
            log.error("系统异常", e);
            response.setRetCode(PlatformCodeEnum.SYS_ERROR.getCode());
            response.setRetMsg(PlatformCodeEnum.SYS_ERROR.getDesc());
            response.setFlag(false);
        }
        return response;
    }

    /**
     * 告警列表
     *
     * @param alertList
     * @return
     * @throws
     * @Author: YangPeng
     * @Date: 2020/1/2 16:46
     */
    @Override
    public BaseResponse<PageDTO<AlertListDTO>> alertList(AlertListQueryDTO alertList, IFWUser user) {
        BaseResponse<PageDTO<AlertListDTO>> response = new BaseResponse<>();
        PageDTO<AlertListDTO> resultPage = new PageDTO<>();
        try {
            String tenantId = CheckAccessUtils.judgeTenantId(alertList.getTenantId(), user);
            alertList.setTenantId(tenantId);
            AlarmRuleListReqBO query = new AlarmRuleListReqBO();
            QueryAppIdQueryDoBO queryAppId = new QueryAppIdQueryDoBO();
            queryAppId.setTenantId(tenantId);
            QueryAppIdBO appid = monitorRepository.queryAppId(queryAppId);
            if (StringUtils.isEmpty(appid.getAppId())) {
                throw new BizException("无appId,请申请appId", PlatformCodeEnum.SYS_ERROR.getCode());
            }
            BeanUtils.copyProperties(alertList, query);
            query.setAppId(appid.getAppId());
            query.setPageSize(alertList.getPageSize());
            query.setPageNum(alertList.getPageIndex());
            PageDTO<AlarmRuleListRespBO> alarmRuleBOS = godzillaServerRepository.queryAlarmRuleList(query);
            if (alarmRuleBOS != null) {
                BeanUtils.copyProperties(alarmRuleBOS, resultPage);
                List<AlertListDTO> resultList = IFWBeanCopyUtil.mapAsList(alarmRuleBOS.getResult(), AlertListDTO.class);
                resultPage.setResult(resultList);
            }
            response.setData(resultPage);
            response.setRetCode(PlatformCodeEnum.EXECUTE_SUCCESS.getCode());
        } catch (BizException bizException) {
            response.setRetCode(bizException.getCode());
            response.setRetMsg(bizException.getMessage());
            response.setFlag(false);
        } catch (Throwable e) {
            log.error("系统异常", e);
            response.setRetCode(PlatformCodeEnum.SYS_ERROR.getCode());
            response.setRetMsg(PlatformCodeEnum.SYS_ERROR.getDesc());
            response.setFlag(false);
        }
        return response;
    }

    /**
     * 编辑告警规则
     *
     * @param updateAlert
     * @return com.ifourthwall.common.base.BaseResponse
     * @throws
     * @Author: YangPeng
     * @Date: 2020/1/2 17:15
     */
    @Override
    public BaseResponse updateAlert(UpAlertNoticeDTO updateAlert, IFWUser user) {
        BaseResponse response = new BaseResponse<>();
        try {
            String tenantId = CheckAccessUtils.judgeTenantId(updateAlert.getTenantId(), user);
            updateAlert.setTenantId(tenantId);
            QueryAppIdQueryDoBO queryAppIdQueryDTO = new QueryAppIdQueryDoBO();
            queryAppIdQueryDTO.setTenantId(tenantId);
            QueryAppIdBO queryAppIdBO = monitorRepository.queryAppId(queryAppIdQueryDTO);
            if (StringUtils.isEmpty(queryAppIdBO.getAppId())) {
                throw new BizException("无appId,请申请appId", PlatformCodeEnum.SYS_ERROR.getCode());
            }
            AlarmRuleInsertBO alarmRuleBO = new AlarmRuleInsertBO();
            BeanUtils.copyProperties(updateAlert, alarmRuleBO);
            alarmRuleBO.setAppId(queryAppIdBO.getAppId());
            AlarmRuleInsertBO insert = new AlarmRuleInsertBO();
            ArrayList<String> ruleList = new ArrayList<>();
            if (!StringUtils.isEmpty(alarmRuleBO.getAlarmChannel())) {
                String[] split = alarmRuleBO.getAlarmChannel().split(",");
                for (String rule : split) {
                    if (!"callback".equals(rule)) {
                        ruleList.add(rule);
                    }
                }
            }
            if (DataUtils.isListAvali(ruleList)){
                alarmRuleBO.setAlarmChannel(String.join(",",ruleList));
            }else{
                alarmRuleBO.setAlarmChannel(null);
            }
            godzillaServerRepository.updateAlarmRule(alarmRuleBO);
            response.setRetCode(PlatformCodeEnum.EXECUTE_SUCCESS.getCode());
        } catch (BizException bizException) {
            response.setRetCode(bizException.getCode());
            response.setRetMsg(bizException.getMessage());
            response.setFlag(false);
        } catch (Throwable e) {
            log.error("系统异常", e);
            response.setRetCode(PlatformCodeEnum.SYS_ERROR.getCode());
            response.setRetMsg(PlatformCodeEnum.SYS_ERROR.getDesc());
            response.setFlag(false);
        }
        return response;
    }

    /**
     * 告警规则详情
     *
     * @param alertInfo
     * @return com.ifourthwall.common.base.BaseResponse<com.ifourthwall.dbm.asset.dto.QueryAlertInfoDTO>
     * @throws
     * @Author: YangPeng
     * @Date: 2020/1/2 17:24
     */
    @Override
    public BaseResponse<QueryAlertInfoDTO> queryAlert(QueryAlertDTO alertInfo, IFWUser user) {
        BaseResponse<QueryAlertInfoDTO> response = new BaseResponse<>();
        try {
            String tenantId = CheckAccessUtils.judgeTenantId(alertInfo.getTenantId(), user);
            QueryAppIdQueryDoBO queryAppIdQueryDTO = new QueryAppIdQueryDoBO();
            queryAppIdQueryDTO.setTenantId(tenantId);
            QueryAppIdBO queryAppIdBO = monitorRepository.queryAppId(queryAppIdQueryDTO);
            QueryAlarmRuleDetailBO query = new QueryAlarmRuleDetailBO();
            query.setAlarmRuleId(alertInfo.getAlarmRuleId());
            AlarmRuleBO alarmRuleBO = godzillaServerRepository.queryAlarmRuleDetail(query);
            QueryAlertInfoDTO result = IFWBeanCopyUtil.map(alarmRuleBO, QueryAlertInfoDTO.class);
            response.setData(result);
            response.setRetCode(PlatformCodeEnum.EXECUTE_SUCCESS.getCode());
        } catch (BizException bizException) {
            response.setRetCode(bizException.getCode());
            response.setRetMsg(bizException.getMessage());
            response.setFlag(false);
        } catch (Throwable e) {
            log.error("系统异常", e);
            response.setRetCode(PlatformCodeEnum.SYS_ERROR.getCode());
            response.setRetMsg(PlatformCodeEnum.SYS_ERROR.getDesc());
            response.setFlag(false);
        }
        return response;
    }

    /**
     * 删除告警规则
     *
     * @param alertInfo
     * @return com.ifourthwall.common.base.BaseResponse
     * @throws
     * @Author: YangPeng
     * @Date: 2020/1/2 17:39
     */
    @Override
    public BaseResponse deleteAlert(QueryAlertDTO alertInfo, IFWUser user) {
        BaseResponse response = new BaseResponse<>();
        try {

            godzillaServerRepository.deleteAlarmRule(alertInfo.getAlarmRuleId());
            response.setRetCode(PlatformCodeEnum.EXECUTE_SUCCESS.getCode());
        } catch (BizException bizException) {
            response.setRetCode(bizException.getCode());
            response.setRetMsg(bizException.getMessage());
            response.setFlag(false);
        } catch (Throwable e) {
            log.error("系统异常", e);
            response.setRetCode(PlatformCodeEnum.SYS_ERROR.getCode());
            response.setRetMsg(PlatformCodeEnum.SYS_ERROR.getDesc());
            response.setFlag(false);
        }
        return response;
    }

    /**
     * 获取AppId
     *
     * @param user
     * @return com.ifourthwall.common.base.BaseResponse
     * @throws
     * @Author: YangPeng
     * @Date: 2020/1/2 17:43
     */
    @Override
    public BaseResponse getAppId(GetAppIdDTO appIdDTO, IFWUser user) {
        BaseResponse response = new BaseResponse<>();
        try {
            String tenantId = CheckAccessUtils.judgeTenantId(appIdDTO.getTenantId(), user);
            QueryAppIdQueryDoBO queryAppIdQueryDTO = new QueryAppIdQueryDoBO();
            //校验租户是否有权限
            ServiceWetherPurchaseReqDTO serviceWetherPurchaseReqDTO = new ServiceWetherPurchaseReqDTO();
            serviceWetherPurchaseReqDTO.setTenantId(tenantId);
            serviceWetherPurchaseReqDTO.setServiceId(sentryServiceId);
            boolean buy = userRepository.queryTenantWhetherPurchaseService(serviceWetherPurchaseReqDTO);
            if (buy) {
                queryAppIdQueryDTO.setTenantId(tenantId);
                QueryAppIdBO queryAppIdBO = monitorRepository.queryAppId(queryAppIdQueryDTO);
                if (queryAppIdBO.getAppId() != null) {
                    throw new BizException("请勿重复创建AppId!", "14");
                }
                AppReqBO creatApp = new AppReqBO();
                creatApp.setAppName("sentry-" + user.getOrganizationName());
                //获取Appid
                AppRespBO app = godzillaServerRepository.createApp(creatApp);
                if (app == null || app.getAppId() == null) {
                    throw new BizException("无法获取appId，请重试!", "15");
                }
                //创建默认告警规则
                AlarmRuleInsertBO insert = new AlarmRuleInsertBO();
                insert.setAppId(app.getAppId());
                insert.setCallbackUrl(callbackUrl);
                insert.setCallbackToken(callbackToken);
                insert.setAlarmChannel("callback");
                insert.setAlarmRuleName("sentry默认告警模板-" + user.getOrganizationName());
                insert.setAlarmCycle("0");
                log.info("addAlarmRule入参：" + insert);
                String ruleId = godzillaServerRepository.addAlarmRule(insert);
                //存入appId
                InsertAppIdQuDoBO insertAppIdQuDoBO = new InsertAppIdQuDoBO();
                insertAppIdQuDoBO.setTenantId(tenantId);
                insertAppIdQuDoBO.setGodzillaAppId(app.getAppId());
                insertAppIdQuDoBO.setCreateTime(new Date());
                insertAppIdQuDoBO.setCreateBy(user.getUserId());
                insertAppIdQuDoBO.setDefaultAlarmRuleId(ruleId);
                monitorRepository.insertAppId(insertAppIdQuDoBO);
            } else {
                throw new BizException(PlatformCodeEnum.TENANT_NOT_BUY_SERVICE.getDesc(), PlatformCodeEnum.TENANT_NOT_BUY_SERVICE.getCode());
            }
            response.setRetCode(PlatformCodeEnum.EXECUTE_SUCCESS.getCode());
        } catch (BizException bizException) {
            response.setRetCode(bizException.getCode());
            response.setRetMsg(bizException.getMessage());
            response.setFlag(false);
        } catch (Throwable e) {
            log.error("系统异常", e);
            response.setRetCode(PlatformCodeEnum.SYS_ERROR.getCode());
            response.setRetMsg(PlatformCodeEnum.SYS_ERROR.getDesc());
            response.setFlag(false);
        }
        return response;
    }

    /**
     * 查询appid
     *
     * @param user
     * @return com.ifourthwall.common.base.BaseResponse
     * @throws
     * @Author: YangPeng
     * @Date: 2020/1/2 20:00
     */
    @Override
    public BaseResponse<QueryAppIdDTO> queryAppId(QueryAppIdQueryDTO query, IFWUser user) {
        String tenantId = CheckAccessUtils.judgeTenantId(query.getTenantId(), user);
        query.setTenantId(tenantId);
        return monitorFacade.queryAppId(query);
    }

    /**
     * 监控指标详情(sentry数据)
     *
     * @param user
     * @return com.ifourthwall.common.base.BaseResponse
     * @throws
     * @Author: YangPeng
     * @Date: 2020/1/3 18:18
     */
    @Override
    public BaseResponse<AlertIndexInfoDTO> sentryInfo(SentryInfoQuBasisDTO sentryInfo, IFWUser user) {
        BaseResponse<AlertIndexInfoDTO> response = new BaseResponse<>();
        try {
            String tenantId = CheckAccessUtils.judgeTenantId(sentryInfo.getTenantId(), user);
            sentryInfo.setTenantId(tenantId);
            //查询监控对象信息
            QueryMonitorInfoQueryBasisDoBO query = new QueryMonitorInfoQueryBasisDoBO();
            query.setAssetId(sentryInfo.getAssetId());
            query.setTenantId(tenantId);
            query.setProjectId(sentryInfo.getProjectId());
            query.setLanguageCode(sentryInfo.getLanguageCode());
            QueryMonitorInfoDoBO queryMonitorInfoDoBOPageDTO = monitorRepository.queryMonitorInfo(query);
            if (queryMonitorInfoDoBOPageDTO == null) {
                return response;
            }
            AlertIndexInfoDTO resultDTO = IFWBeanCopyUtil.map(queryMonitorInfoDoBOPageDTO, AlertIndexInfoDTO.class);
            //查询告警记录
            QueryRecordingQueryDoBO queryRecord = new QueryRecordingQueryDoBO();
            queryRecord.setProjectId(sentryInfo.getProjectId());
            queryRecord.setMonitorTargetId(sentryInfo.getMonitorTargetId());
            queryRecord.setLanguageCode("CN");
            queryRecord.setTenantId(tenantId);
            PageDTO<QueryRecordingListDoBO> monitorPage = monitorRepository.queryRecordingList(queryRecord);
            if (!DataUtils.isListAvali(monitorPage.getResult())) {
                response.setData(resultDTO);
                return response;
            }
            List<AlertIndexListDTO> alertIndexListDTOS = IFWBeanCopyUtil.mapAsList(monitorPage.getResult(), AlertIndexListDTO.class);
            resultDTO.setAlertRecording(alertIndexListDTOS);
            response.setData(resultDTO);
            response.setRetCode(PlatformCodeEnum.EXECUTE_SUCCESS.getCode());
        } catch (BizException bizException) {
            response.setRetCode(bizException.getCode());
            response.setRetMsg(bizException.getMessage());
            response.setFlag(false);
        } catch (Throwable e) {
            log.error("系统异常", e);
            response.setRetCode(PlatformCodeEnum.SYS_ERROR.getCode());
            response.setRetMsg(PlatformCodeEnum.SYS_ERROR.getDesc());
            response.setFlag(false);
        }
        return response;
    }

    /**
     * 监控指标详情(哥斯拉数据)
     *
     * @param godzillaInfoQu
     * @param user
     * @return com.ifourthwall.common.base.BaseResponse<com.ifourthwall.dbm.asset.dto.GodzillaInfoDTO>
     * @throws
     * @Author: YangPeng
     * @Date: 2020/1/4 18:14
     */
    @Override
    public BaseResponse<MetricValueRecordListDTO> godzillaInfo(GodzillaInfoQuDTO godzillaInfoQu, IFWUser user) {
        BaseResponse<MetricValueRecordListDTO> response = new BaseResponse<>();
        try {
            String tenantId = CheckAccessUtils.judgeTenantId(godzillaInfoQu.getTenantId(), user);
            godzillaInfoQu.setTenantId(tenantId);
            QueryAppIdQueryDoBO queryAppId = new QueryAppIdQueryDoBO();
            queryAppId.setTenantId(tenantId);
            QueryAppIdBO queryAppIdBO = monitorRepository.queryAppId(queryAppId);
            if (queryAppIdBO == null) {
                throw new BizException("不存在appId，请申请!", "15");
            }
            List<GodzillaInfoDTO> resultList = new ArrayList<>();
            MetricValueRecordListReqBO querygodzilla = new MetricValueRecordListReqBO();
            BeanUtils.copyProperties(godzillaInfoQu, querygodzilla);
            querygodzilla.setAppId(queryAppIdBO.getAppId());
            MetricValueRecordListBO queryInfo = godzillaServerRepository.queryMetricValueRecordList(querygodzilla);
            if (queryInfo != null) {
                MetricValueRecordListDTO result = IFWBeanCopyUtil.map(queryInfo, MetricValueRecordListDTO.class);
                QueryRuleAndStartBO queryRuleAndStartBO = new QueryRuleAndStartBO();
                queryRuleAndStartBO.setDataPointId(godzillaInfoQu.getDataPointId());
                QueryRuleAndStartDoBO queryRuleAndStartDoBO = godzillaServerRepository.queryRuleAndStart(queryRuleAndStartBO);
                if (queryRuleAndStartDoBO != null) {
                    result.setUnit(queryRuleAndStartDoBO.getUnit());
                }
                response.setData(result);
            }
            response.setRetCode(PlatformCodeEnum.EXECUTE_SUCCESS.getCode());
        } catch (BizException bizException) {
            response.setRetCode(bizException.getCode());
            response.setRetMsg(bizException.getMessage());
            response.setFlag(false);
        } catch (Throwable e) {
            log.error("系统异常", e);
            response.setRetCode(PlatformCodeEnum.SYS_ERROR.getCode());
            response.setRetMsg(PlatformCodeEnum.SYS_ERROR.getDesc());
            response.setFlag(false);
        }
        return response;
    }

    /**
     * sentry统计信息
     *
     * @param sentryInfo
     * @param user
     * @return
     */
    @Override
    public BaseResponse<QuerySentryInfoDTO> querySentryInfo(QuerySentryInfoQueryDTO sentryInfo, IFWUser user) {
        String tenantId = CheckAccessUtils.judgeTenantId(sentryInfo.getTenantId(), user);
        sentryInfo.setTenantId(tenantId);
        return monitorFacade.querySentryInfo(sentryInfo);
    }

    /**
     * sentry统计信息(设备统计)
     *
     * @param sentryInfo
     * @return
     */
    @Override
    public BaseResponse<QuerySentryInfoAssetDTO> querySentryInfoAsset(QuerySentryInfoQueryDTO sentryInfo, IFWUser user) {
        String tenantId = CheckAccessUtils.judgeTenantId(sentryInfo.getTenantId(), user);
        sentryInfo.setTenantId(tenantId);
        return monitorFacade.querySentryInfoAsset(sentryInfo);
    }

    /**
     * sentry统计信息（7日告警）
     *
     * @param sentryInfo
     * @return
     */
    @Override
    public BaseResponse<QuerySentryInfoSevenDayDTO> querySentryInfoSevenDay(QuerySentryInfoQueryDTO sentryInfo, IFWUser user) {
        String tenantId = CheckAccessUtils.judgeTenantId(sentryInfo.getTenantId(), user);
        sentryInfo.setTenantId(tenantId);
        return monitorFacade.querySentryInfoSevenDay(sentryInfo);
    }

    /**
     * sentry统计信息（告警排行）
     *
     * @param sentryInfo
     * @return
     */
    @Override
    public BaseResponse<SentryInfoRankingDTO> querySentryInfoRanking(QuerySentryInfoQueryDTO sentryInfo, IFWUser user) {
        String tenantId = CheckAccessUtils.judgeTenantId(sentryInfo.getTenantId(), user);
        sentryInfo.setTenantId(tenantId);
        return monitorFacade.querySentryInfoRanking(sentryInfo);
    }

    /**
     * sentry统计信息（设备完好率）
     *
     * @param sentryInfo
     * @return
     */
    @Override
    public BaseResponse<SentryInfoIntactDTO> querySentryInfoIntact(QuerySentryInfoQueryDTO sentryInfo, IFWUser user) {
        String tenantId = CheckAccessUtils.judgeTenantId(sentryInfo.getTenantId(), user);
        sentryInfo.setTenantId(tenantId);
        return monitorFacade.querySentryInfoIntact(sentryInfo);
    }

    /**
     * seer系统统计列表
     *
     * @param sentryInfo
     * @return
     */
    @Override
    public BaseResponse<IFWPageInfo<SeerTagListDTO>> seerTagList(QuerySentryInfoQueryDTO sentryInfo , IFWUser user) {
        String tenantId = CheckAccessUtils.judgeTenantId(sentryInfo.getTenantId(), user);
        sentryInfo.setTenantId(tenantId);
        return monitorFacade.seerTagList(sentryInfo);
    }

    /**
     * sentry实时监控图-检测设备总数
     * 每3小时调用一次
     *
     * @param rankingQuDTO
     * @param user
     * @return com.ifourthwall.common.base.BaseResponse<com.ifourthwall.dbm.asset.dto.sentryMonitorRankingDTO>
     * @throws
     * @Author: YangPeng
     * @Date: 2020/3/9 10:11
     */
    @Override
    public BaseResponse<SentryMonitorRankingDTO> sentryMonitor(SentryMonitorRankingQuDTO rankingQuDTO, IFWUser user) {
        BaseResponse<SentryMonitorRankingDTO> response = new BaseResponse<>();
        SentryMonitorRankingDTO result = new SentryMonitorRankingDTO();
        try {
//            String key = null;
//            //查询redis中是否存在该数据
//            if (!StringUtils.isEmpty(rankingQuDTO.getSpaceId())) {
//                //seer端通过空间筛选（项目id + 查询周期 + 空间id）
//                key = RedisKeyPrefixUtil.getAssetCountInfoType(rankingQuDTO.getProjectId() + rankingQuDTO.getTime() + rankingQuDTO.getSpaceId());
//            }
//            if (StringUtils.isEmpty(rankingQuDTO.getSpaceId())) {
//                //web端普通筛选
//                key = RedisKeyPrefixUtil.getAssetCountInfoType(rankingQuDTO.getProjectId() + rankingQuDTO.getTime());
//            }
//            SentryMonitorRankingDTO queryRedis = (SentryMonitorRankingDTO) redisTemplate.opsForValue().get(key);
//            if (queryRedis == null) {
            Date startTime = null;
            Date endTime = null;
            SimpleDateFormat formatter = new SimpleDateFormat("yyyy-MM-dd HH:mm:ss");
            if ("1".equals(rankingQuDTO.getTime())) {
                startTime = DataUtils.getDayBegin();
                endTime = DataUtils.getDayEnd();
            }
            if ("2".equals(rankingQuDTO.getTime())) {
                startTime = DataUtils.getBeginDayOfWeek();
                endTime = DataUtils.getEndDayOfWeek();
            }
            if ("3".equals(rankingQuDTO.getTime())) {
                startTime = DataUtils.getBeginDayOfMonth();
                endTime = DataUtils.getEndDayOfMonth();
            }
            //如果过是需要通过空间id查询
            ArrayList<String> spaceIds = new ArrayList<>();
            if (!StringUtils.isEmpty(rankingQuDTO.getSpaceId())) {
                GetSonSpaceQueryBO querySpace = new GetSonSpaceQueryBO();
                querySpace.setSpaceId(rankingQuDTO.getSpaceId());
                GetSonSpaceBO sonpace = spaceRepository.getSonpace(querySpace);
                spaceIds.addAll(sonpace.getSpaceId());
            }
            String tenantId = CheckAccessUtils.judgeTenantId(rankingQuDTO.getTenantId(), user);
            rankingQuDTO.setTenantId(tenantId);
            QueryMonitorListDoQueryBO queryMonitorListDoQueryBO = new QueryMonitorListDoQueryBO();
            queryMonitorListDoQueryBO.setProjectId(rankingQuDTO.getProjectId());
            queryMonitorListDoQueryBO.setTenantId(tenantId);
            queryMonitorListDoQueryBO.setLanguageCode(rankingQuDTO.getLanguageCode());
            if (!StringUtils.isEmpty(rankingQuDTO.getSpaceId())) {
                queryMonitorListDoQueryBO.setSpaceId(spaceIds);
            }
            //查询检测设备总数
            IFWPageInfo<QueryMonitorListDoBO> monitorList = monitorRepository.queryMonitorList(queryMonitorListDoQueryBO);
            if (monitorList != null) {
                QueryRecordingQueryDoBO queryRecordingQueryDoBO = new QueryRecordingQueryDoBO();
                queryRecordingQueryDoBO.setProjectId(rankingQuDTO.getProjectId());
                queryRecordingQueryDoBO.setLanguageCode(rankingQuDTO.getLanguageCode());
                queryRecordingQueryDoBO.setTenantId(tenantId);
                queryRecordingQueryDoBO.setBeginTime(formatter.format(startTime));
                queryRecordingQueryDoBO.setEndTime(formatter.format(endTime));
                if (!StringUtils.isEmpty(rankingQuDTO.getSpaceId())) {
                    queryRecordingQueryDoBO.setSpaceId(spaceIds);
                }
                //计算设备完好率及设备故障率
                Integer faultNumber = 0;
                Integer normalNumber = 0;
                ArrayList<QueryMonitorListDoBO> faultAssetList = new ArrayList<>();
                ArrayList<QueryMonitorListDoBO> normalAssetList = new ArrayList<>();
                Map<String, List<QueryMonitorListDoBO>> assetStatusMap = monitorList.getList().stream().filter(a -> a.getAssetStatus() != null).collect(Collectors.groupingBy(QueryMonitorListDoBO::getAssetStatus));
                for (Map.Entry<String, List<QueryMonitorListDoBO>> assetStatus : assetStatusMap.entrySet()) {
                    if ("1".equals(assetStatus.getKey())) {
                        //正常率
                        normalNumber = assetStatus.getValue().size();
                        normalAssetList.addAll(assetStatus.getValue());
                    }
                    if ("2".equals(assetStatus.getKey())) {
                        //故障率
                        faultNumber = assetStatus.getValue().size();
                        //故障的检测设备
                        faultAssetList.addAll(assetStatus.getValue());
                    }
                }
                result.setAssetNumber(monitorList.getList().size());
                if (faultNumber == 0) {
                    //不存在告警，故障率就为0，完好率为100
                    result.setNormalProportion(new BigDecimal("100"));
                    result.setFaultProportion(new BigDecimal("0"));
                    result.setAssetFaultedNumber(0);
                } else {
                    result.setAssetFaultedNumber(faultNumber);
                    result.setFaultProportion(MathUtils.divideGetPercent(faultNumber, monitorList.getList().size(), 2, RoundingMode.HALF_UP));
                    result.setNormalProportion(MathUtils.divideGetPercent(normalNumber, monitorList.getList().size(), 2, RoundingMode.HALF_UP));
                }
                //按照时间段查询故障过的设备（记录）
                PageDTO<QueryRecordingListDoBO> recordingList1 = monitorRepository.queryRecordingList(queryRecordingQueryDoBO);
                if (DataUtils.isListAvali(recordingList1.getResult())) {
                    //故障修复数量
                    Integer faultReNumber = 0;
                    //按监控对象产来区分
                    Map<String, List<QueryRecordingListDoBO>> monitorTargetMap = recordingList1.getResult().stream().collect(Collectors.groupingBy(QueryRecordingListDoBO::getMonitorTargetId));
                    List<String> monitorTargetList = recordingList1.getResult().stream().map(QueryRecordingListDoBO::getMonitorTargetId).distinct().collect(Collectors.toList());
                    for (QueryMonitorListDoBO monitorListDoBO : normalAssetList) {
                        if (monitorTargetList.contains(monitorListDoBO.getMonitorTargetId())) {
                            faultReNumber++;
                        }
                    }
                    result.setRepairProportion(MathUtils.divideGetPercent(faultReNumber, monitorTargetList.size(), 2, RoundingMode.HALF_UP));
                } else {
                    //时间段内没有故障过的设备
                    result.setRepairProportion(new BigDecimal("0"));
                }

            }
            response.setData(result);
            response.setRetCode(PlatformCodeEnum.EXECUTE_SUCCESS.getCode());
        } catch (BizException bizException) {
            response.setRetCode(bizException.getCode());
            response.setRetMsg(bizException.getMessage());
            response.setFlag(false);
        } catch (Throwable e) {
            log.error("系统异常", e);
            response.setRetCode(PlatformCodeEnum.SYS_ERROR.getCode());
            response.setRetMsg(PlatformCodeEnum.SYS_ERROR.getDesc());
            response.setFlag(false);
        }

        return response;
    }


    /**
     * sentry实时监控图-故障排行
     *
     * @param rankingQuDTO
     * @param user
     * @return com.ifourthwall.common.base.BaseResponse<com.ifourthwall.dbm.asset.dto.sentryRankDTO>
     * @throws
     * @Author: YangPeng
     * @Date: 2020/3/12 17:46
     */
    @Override
    public BaseResponse<SentryRankDTO> sentryRanking(SentryMonitorRankingQuDTO rankingQuDTO, IFWUser user) {
        BaseResponse<SentryRankDTO> response = new BaseResponse<>();
        SentryRankDTO result = new SentryRankDTO();
        ArrayList<MonitorRankingDTO> resultList = new ArrayList<>();

        try {
            String key = null;
            //查询redis中是否存在该数据
            if (!StringUtils.isEmpty(rankingQuDTO.getSpaceId())) {
                //seer端通过空间筛选（项目id + 查询周期 + 空间id）
                key = RedisKeyPrefixUtil.getAssetCountInfoType(rankingQuDTO.getProjectId() + rankingQuDTO.getTime() + rankingQuDTO.getSpaceId() + rankingQuDTO.getLanguageCode());
            }
            if (StringUtils.isEmpty(rankingQuDTO.getSpaceId())) {
                //web端普通筛选
                key = RedisKeyPrefixUtil.getAssetCountInfoType(rankingQuDTO.getProjectId() + rankingQuDTO.getTime() + rankingQuDTO.getLanguageCode());
            }
            //查询redis中是否存在该数据
//            SentryRankDTO queryRedis = (SentryRankDTO) redisTemplate.opsForValue().get(RedisKeyPrefixUtil.getRankingCountInfoType(key));
//            if (queryRedis == null) {
            Date startTime = null;
            Date endTime = null;
            SimpleDateFormat formatter = new SimpleDateFormat("yyyy-MM-dd HH:mm:ss");
            if ("1".equals(rankingQuDTO.getTime())) {
                startTime = DataUtils.getDayBegin();
                endTime = DataUtils.getDayEnd();
            }
            if ("2".equals(rankingQuDTO.getTime())) {
                startTime = DataUtils.getBeginDayOfWeek();
                endTime = DataUtils.getEndDayOfWeek();
            }
            if ("3".equals(rankingQuDTO.getTime())) {
                startTime = DataUtils.getBeginDayOfMonth();
                endTime = DataUtils.getEndDayOfMonth();
            }
            String tenantId = CheckAccessUtils.judgeTenantId(rankingQuDTO.getTenantId(), user);
            rankingQuDTO.setTenantId(tenantId);
            //查询系统总数
            TranslationDoBO translationDoBO = new TranslationDoBO();
            translationDoBO.setLanguageCode(rankingQuDTO.getLanguageCode());
            AllTranslationDoBO allTranslationDoBO = assetRepository.allTranslation(translationDoBO);
            for (Map.Entry<String, String> tagIds : allTranslationDoBO.getSystem().entrySet()) {
                MonitorRankingDTO resultDTO = new MonitorRankingDTO();
                resultDTO.setTagName(tagIds.getValue());
                resultDTO.setTagId(tagIds.getKey());
                resultList.add(resultDTO);
            }
            result.setRankInfo(resultList);
            //如果过是需要通过空间id查询
            ArrayList<String> spaceIds = new ArrayList<>();
            if (!StringUtils.isEmpty(rankingQuDTO.getSpaceId())) {
                GetSonSpaceQueryBO querySpace = new GetSonSpaceQueryBO();
                querySpace.setSpaceId(rankingQuDTO.getSpaceId());
                GetSonSpaceBO sonpace = spaceRepository.getSonpace(querySpace);
                spaceIds.addAll(sonpace.getSpaceId());
            }

            //查询监控总数
            QueryMonitorListDoQueryBO queryMonitorListDoQueryBO = new QueryMonitorListDoQueryBO();
            queryMonitorListDoQueryBO.setProjectId(rankingQuDTO.getProjectId());
            queryMonitorListDoQueryBO.setTenantId(tenantId);
            queryMonitorListDoQueryBO.setLanguageCode(rankingQuDTO.getLanguageCode());
            if (!StringUtils.isEmpty(rankingQuDTO.getSpaceId())) {
                queryMonitorListDoQueryBO.setSpaceId(spaceIds);
            }
            //查询检测设备总数
            IFWPageInfo<QueryMonitorListDoBO> monitorList = monitorRepository.queryMonitorList(queryMonitorListDoQueryBO);
            //查询故障总数
            queryMonitorListDoQueryBO.setAssetStatus("2");
            IFWPageInfo<QueryMonitorListDoBO> monitorList1 = monitorRepository.queryMonitorList(queryMonitorListDoQueryBO);
            if (monitorList != null) {
                //查询正在故障的设备
                List<QueryMonitorListDoBO> faultNumber = null;
                //set总数
                if (monitorList1 != null) {
                    faultNumber = monitorList1.getList();
                    result.setTotal(monitorList1.getList().size());
                } else {
                    result.setTotal(0);
                }
                //按照资产系统查询
                List<QueryMonitorListDoBO> notNullList = monitorList.getList().stream().filter(a -> a.getAssetSystemId() != null).collect(Collectors.toList());
                if (DataUtils.isListAvali(notNullList)) {
                    //筛选出有系统id的监控对象，与系统对比
                    Map<String, List<QueryMonitorListDoBO>> tagIdMap = notNullList.stream().collect(Collectors.groupingBy(QueryMonitorListDoBO::getAssetSystemId));
                    for (MonitorRankingDTO resultDTO : result.getRankInfo()) {
                        for (Map.Entry<String, List<QueryMonitorListDoBO>> tagId : tagIdMap.entrySet()) {
                            if (StringUtils.equals(resultDTO.getTagId(), tagId.getKey())) {
                                //查询该系统下设备总数
                                //该系统下有监控对象
                                if (DataUtils.isListAvali(tagId.getValue())) {
                                    //完好设备数量
                                    int normal = 0;
                                    //查询该系统下的故障中和故障过设备
                                    QueryRecordingQueryDoBO queryRecording = new QueryRecordingQueryDoBO();
                                    queryRecording.setBeginTime(formatter.format(startTime));
                                    queryRecording.setEndTime(formatter.format(endTime));
                                    queryRecording.setProjectId(rankingQuDTO.getProjectId());
                                    queryRecording.setLanguageCode(rankingQuDTO.getLanguageCode());
                                    queryRecording.setTenantId(tenantId);
                                    queryRecording.setAssetSystemId(tagId.getKey());
                                    if (!StringUtils.isEmpty(rankingQuDTO.getSpaceId())) {
                                        queryRecording.setSpaceId(spaceIds);
                                    }
                                    PageDTO<QueryRecordingListDoBO> recordingList = monitorRepository.queryRecordingList(queryRecording);
                                    //当监控对象发送过告警时
                                    if (DataUtils.isListAvali(recordingList.getResult())) {
                                        List<String> ids = recordingList.getResult().stream().map(QueryRecordingListDoBO::getAssetId).distinct().collect(Collectors.toList());
                                        for (QueryMonitorListDoBO monitor : tagId.getValue()) {
                                            if (!ids.contains(monitor.getAssetId())) {
                                                normal++;
                                            }
                                        }
                                        resultDTO.setNormalNumber(normal);
                                        //存在正在故障的设备
                                        if (DataUtils.isListAvali(faultNumber)) {
                                            resultDTO.setFaultNumber(faultNumber.size());
                                            resultDTO.setFixNumber(tagId.getValue().size() - faultNumber.size() - normal);
                                        } else {
                                            resultDTO.setFaultNumber(0);
                                            //不存在故障中的设备但是有故障过（故障过和故障中的设备）
                                            resultDTO.setFixNumber(tagId.getValue().size() - normal);
                                        }
                                    } else {
                                        //没有发生过告警的时候
                                        resultDTO.setFaultNumber(0);
                                        resultDTO.setFixNumber(0);
                                        resultDTO.setNormalNumber(tagId.getValue().size());
                                    }
                                }

                            }
                        }
                    }
                }
            }
            response.setData(result);
//                    redisTemplate.opsForValue().set(RedisKeyPrefixUtil.getRankingCountInfoType(key), result, 180, TimeUnit.MINUTES);
//                }

//            if (queryRedis != null) {
//                response.setData(queryRedis);
//            }
            response.setRetCode(PlatformCodeEnum.EXECUTE_SUCCESS.getCode());
        } catch (BizException bizException) {
            response.setRetCode(bizException.getCode());
            response.setRetMsg(bizException.getMessage());
            response.setFlag(false);
        } catch (Throwable e) {
            log.error("系统异常", e);
            response.setRetCode(PlatformCodeEnum.SYS_ERROR.getCode());
            response.setRetMsg(PlatformCodeEnum.SYS_ERROR.getDesc());
            response.setFlag(false);
        }
        return response;
    }

    /**
     * sentry实时监控图-系统故障
     *
     * @param tagQuery
     * @param user
     * @return com.ifourthwall.common.base.BaseResponse<java.util.List < com.ifourthwall.dbm.asset.dto.TagInfoDTO>>
     * @throws
     * @Author: YangPeng
     * @Date: 16:42
     */
    @Override
    public BaseResponse<List<TagInfoDTO>> tagInfo(TagInfoQuDTO tagQuery, IFWUser user) {
        BaseResponse<List<TagInfoDTO>> response = new BaseResponse<>();
        try {
            String tenantId = CheckAccessUtils.judgeTenantId(tagQuery.getTenantId(), user);
            tagQuery.setTenantId(tenantId);
            //查询监控总数
            QueryMonitorListDoQueryBO queryMonitorListDoQueryBO = new QueryMonitorListDoQueryBO();
            queryMonitorListDoQueryBO.setProjectId(tagQuery.getProjectId());
            queryMonitorListDoQueryBO.setTenantId(tenantId);
            queryMonitorListDoQueryBO.setLanguageCode(tagQuery.getLanguageCode());
            //查询检测设备总数
            IFWPageInfo<QueryMonitorListDoBO> monitorList = monitorRepository.queryMonitorList(queryMonitorListDoQueryBO);
            ArrayList<TagInfoDTO> resultList = new ArrayList<>();
            if (monitorList != null) {
                //筛选出有系统的检测设备
                List<QueryMonitorListDoBO> notNullList = monitorList.getList().stream().filter(monitor -> monitor.getAssetSystemId() != null).collect(Collectors.toList());
                if (DataUtils.isListAvali(notNullList)) {
                    Map<String, List<QueryMonitorListDoBO>> tagMap = notNullList.stream().collect(Collectors.groupingBy(QueryMonitorListDoBO::getAssetSystemId));
                    for (Map.Entry<String, List<QueryMonitorListDoBO>> tagId : tagMap.entrySet()) {
                        TagInfoDTO result = new TagInfoDTO();
                        result.setTagId(tagId.getKey());
                        result.setTagName(tagId.getValue().get(0).getSystemName());
                        int normalNumber = 0;
                        int abnormalNumber = 0;
                        Map<String, List<QueryMonitorListDoBO>> assetStatusMap = tagId.getValue().stream().filter(a -> a.getAssetStatus() != null).collect(Collectors.groupingBy(QueryMonitorListDoBO::getAssetStatus));
                        for (Map.Entry<String, List<QueryMonitorListDoBO>> assetStatus : assetStatusMap.entrySet()) {
                            if ("1".equals(assetStatus.getKey())) {
                                normalNumber = assetStatus.getValue().size();
                            }
                            if ("2".equals(assetStatus.getKey())) {
                                abnormalNumber = assetStatus.getValue().size();
                            }
                        }
                        result.setAbnormalNumber(abnormalNumber);
                        result.setNormalNumber(normalNumber);
                        resultList.add(result);
                    }
                }
            }
            response.setData(resultList);
            response.setRetCode(PlatformCodeEnum.EXECUTE_SUCCESS.getCode());
        } catch (BizException bizException) {
            response.setRetCode(bizException.getCode());
            response.setRetMsg(bizException.getMessage());
            response.setFlag(false);
        } catch (Throwable e) {
            log.error("系统异常", e);
            response.setRetCode(PlatformCodeEnum.SYS_ERROR.getCode());
            response.setRetMsg(PlatformCodeEnum.SYS_ERROR.getDesc());
            response.setFlag(false);
        }
        return response;
    }

    /**
     * sentry实时监控图-系统故障百分比
     *
     * @param percentQuery
     * @param user
     * @return
     * @throws
     * @Author: YangPeng
     * @Date: 2020/3/17 17:08
     */
    @Override
    public BaseResponse<TagFaultPercentDTO> tagFaultPercent(TagFaultPercentQuDTO percentQuery, IFWUser user) {
        BaseResponse<TagFaultPercentDTO> response = new BaseResponse<>();
        try {
            String tenantId = CheckAccessUtils.judgeTenantId(percentQuery.getTenantId(), user);
            percentQuery.setTenantId(tenantId);
            //查询该系统的监控总数
            QueryMonitorListDoQueryBO queryMonitorListDoQueryBO = new QueryMonitorListDoQueryBO();
            queryMonitorListDoQueryBO.setProjectId(percentQuery.getProjectId());
            queryMonitorListDoQueryBO.setTenantId(tenantId);
            queryMonitorListDoQueryBO.setLanguageCode(percentQuery.getLanguageCode());
            queryMonitorListDoQueryBO.setAssetSystemId(percentQuery.getTagId());
            //查询检测设备总数
            IFWPageInfo<QueryMonitorListDoBO> monitorList = monitorRepository.queryMonitorList(queryMonitorListDoQueryBO);
            if (monitorList != null) {
                TagFaultPercentDTO result = new TagFaultPercentDTO();
                result.setTagId(monitorList.getList().get(0).getAssetSystemId());
                result.setTagName(monitorList.getList().get(0).getSystemName());
                result.setAssetNumber(monitorList.getList().size());
                int norma = 0;
                for (QueryMonitorListDoBO monitorListDoBO : monitorList.getList()) {
                    if (!"2".equals(monitorListDoBO.getAssetStatus())) {
                        norma++;
                    }
                }
                BigDecimal oneHundred = new BigDecimal("100");
                BigDecimal normaBig = new BigDecimal(norma);
                BigDecimal normaPercent = MathUtils.divideGetPercent(normaBig, monitorList.getList().size(), 2, RoundingMode.HALF_UP);
                result.setFaultPercent(oneHundred.subtract(normaPercent));
                result.setNormalPercent(normaPercent);
                response.setData(result);
            }
            response.setRetCode(PlatformCodeEnum.EXECUTE_SUCCESS.getCode());
        } catch (BizException bizException) {
            response.setRetCode(bizException.getCode());
            response.setRetMsg(bizException.getMessage());
            response.setFlag(false);
        } catch (Throwable e) {
            log.error("系统异常", e);
            response.setRetCode(PlatformCodeEnum.SYS_ERROR.getCode());
            response.setRetMsg(PlatformCodeEnum.SYS_ERROR.getDesc());
            response.setFlag(false);
        }
        return response;
    }

    /**
     * sentry实时监控图-空间筛选
     *
     * @param spaceQu
     * @param user
     * @return com.ifourthwall.common.base.BaseResponse<java.util.List < com.ifourthwall.dbm.asset.dto.MonitorSpaceDTO>>
     * @throws
     * @Author: YangPeng
     * @Date: 2020/3/18 11:17
     */
    @Override
    public BaseResponse<List<MonitorSpaceDTO>> monitorSpace(MonitorSpaceQuDTO spaceQu, IFWUser user) {
        BaseResponse<List<MonitorSpaceDTO>> response = new BaseResponse<>();
        try {
            String tenantId = CheckAccessUtils.judgeTenantId(spaceQu.getTenantId(), user);
            spaceQu.setTenantId(tenantId);
            //查询该系统的监控总数
            QueryMonitorListDoQueryBO queryMonitorListDoQueryBO = new QueryMonitorListDoQueryBO();
            queryMonitorListDoQueryBO.setProjectId(spaceQu.getProjectId());
            queryMonitorListDoQueryBO.setTenantId(tenantId);
            queryMonitorListDoQueryBO.setLanguageCode(spaceQu.getLanguageCode());
            if (!StringUtils.isEmpty(spaceQu.getSpaceId())) {
                ArrayList<String> ids = new ArrayList<>();
                ids.add(spaceQu.getSpaceId());
                queryMonitorListDoQueryBO.setSpaceId(ids);
            }
            //查询检测设备总数
            IFWPageInfo<QueryMonitorListDoBO> monitorList = monitorRepository.queryMonitorList(queryMonitorListDoQueryBO);
            if (monitorList != null) {
                ArrayList<MonitorSpaceDTO> resultList = new ArrayList<>();
                ArrayList<String> spaceIds = new ArrayList<>();
                Map<String, List<QueryMonitorListDoBO>> spaceMap = monitorList.getList().stream().collect(Collectors.groupingBy(QueryMonitorListDoBO::getProjectSpaceId));
                //按照空间区分
                for (Map.Entry<String, List<QueryMonitorListDoBO>> spaceId : spaceMap.entrySet()) {
                    spaceIds.add(spaceId.getKey());
                    MonitorSpaceDTO result = new MonitorSpaceDTO();
                    result.setSpaceId(spaceId.getKey());
                    result.setAssetNumber(spaceId.getValue().size());
                    int norma = 0;
                    List<QueryMonitorListDoBO> collect = spaceId.getValue().stream().filter(a -> !"2".equals(a.getAssetStatus())).collect(Collectors.toList());
                    if (DataUtils.isListAvali(collect)) {
                        norma = collect.size();
                    }
                    result.setAbnormalNumber(spaceId.getValue().size() - norma);
                    result.setNormalNumber(norma);
                    BigDecimal normalPercent = MathUtils.divideGetPercent(norma, spaceId.getValue().size(), 2, RoundingMode.HALF_UP);
                    result.setNormalPercent(normalPercent);
                    BigDecimal bigDecimal = new BigDecimal("100");
                    result.setAbnormalPercent(bigDecimal.subtract(normalPercent));
                    resultList.add(result);
                }
                //查询空间名称
                GetNewPathBatchQueryBO queryName = new GetNewPathBatchQueryBO();
                queryName.setProjectSpaceId(spaceIds);
                queryName.setProjectId(spaceQu.getProjectId());
                queryName.setLanguageCode(spaceQu.getLanguageCode());
                queryName.setTenantId(spaceQu.getTenantId());
                List<ProjectSpacePathBO> spaceName = spaceRepository.getNewPathBatch(queryName);
                if (DataUtils.isListAvali(spaceName)) {
                    for (MonitorSpaceDTO result : resultList) {
                        for (ProjectSpacePathBO name : spaceName) {
                            if (StringUtils.equals(result.getSpaceId(), name.getId())) {
                                result.setSpaceName(name.getNameStr());
                            }
                        }
                    }
                }
                response.setData(resultList);
            }
            response.setRetCode(PlatformCodeEnum.EXECUTE_SUCCESS.getCode());
        } catch (BizException bizException) {
            response.setRetCode(bizException.getCode());
            response.setRetMsg(bizException.getMessage());
            response.setFlag(false);
        } catch (Throwable e) {
            log.error("系统异常", e);
            response.setRetCode(PlatformCodeEnum.SYS_ERROR.getCode());
            response.setRetMsg(PlatformCodeEnum.SYS_ERROR.getDesc());
            response.setFlag(false);
        }
        return response;
    }

    /**
     * 新增监控对象标签
     *
     * @param insertTag
     * @param user
     * @return
     * @throws
     * @Author: YangPeng
     * @Date: 2020/4/1 9:41
     */
    @Override
    public BaseResponse insertTag(InsertTagQuDTO insertTag, IFWUser user) {
        String tenantId = CheckAccessUtils.judgeTenantId(insertTag.getTenantId(), user);
        insertTag.setTenantId(tenantId);
        UpInsertTagQuDTO upInsertTagQuDTO = IFWBeanCopyUtil.map(insertTag, UpInsertTagQuDTO.class);
        upInsertTagQuDTO.setCreateBy(user.getUserId());
        upInsertTagQuDTO.setType(1);
        return monitorFacade.upInsertTag(upInsertTagQuDTO);
    }

    /**
     * 修改监控对象标签
     *
     * @param updateTag
     * @param user
     * @return com.ifourthwall.common.base.BaseResponse
     * @throws
     * @Author: YangPeng
     * @Date: 2020/4/1 9:42
     */
    @Override
    public BaseResponse updateTag(UpdateTagQuDTO updateTag, IFWUser user) {
        String tenantId = CheckAccessUtils.judgeTenantId(updateTag.getTenantId(), user);
        updateTag.setTenantId(tenantId);
        UpInsertTagQuDTO upInsertTagQuDTO = IFWBeanCopyUtil.map(updateTag, UpInsertTagQuDTO.class);
        upInsertTagQuDTO.setUpdateBy(user.getUserId());
        upInsertTagQuDTO.setType(2);
        return monitorFacade.upInsertTag(upInsertTagQuDTO);
    }

    /**
     * 删除监控对象标签
     *
     * @param deleteTag
     * @param user
     * @return
     * @throws
     * @Author: YangPeng
     * @Date: 2020/4/1 9:42
     */
    @Override
    public BaseResponse deleteTag(DeleteTagQuDTO deleteTag, IFWUser user) {
        String tenantId = CheckAccessUtils.judgeTenantId(deleteTag.getTenantId(), user);
        deleteTag.setTenantId(tenantId);
        deleteTag.setUpdateBy(user.getUserId());
        return monitorFacade.deleteTag(deleteTag);
    }

    /**
     * 查询调用授权数量
     *
     * @param appId
     * @param user
     * @return com.ifourthwall.common.base.BaseResponse<com.ifourthwall.dbm.asset.dto.TransferNumberDTO>
     * @throws
     * @Author: YangPeng
     * @Date: 2020/4/1 11:38
     */
    @Override
    public BaseResponse<TransferNumberDTO> transferNumber(TransferNumberQuDTO appId, IFWUser user) {
        BaseResponse<TransferNumberDTO> response = new BaseResponse<>();
        try {
            String tenantId = CheckAccessUtils.judgeTenantId(appId.getTenantId(), user);
            appId.setTenantId(tenantId);
            TransferNumberQuDoBO transferNumberQuDoBO = new TransferNumberQuDoBO();
            TransferNumberDoBO query = godzillaServerRepository.transferNumber(IFWBeanCopyUtil.map(appId, TransferNumberQuDoBO.class));
            TransferNumberDTO result = IFWBeanCopyUtil.map(query, TransferNumberDTO.class);
            result.setApiLastNumber(query.getApiNumber() - query.getApiUseNumber());
            result.setCallbackLastNumber(query.getCallbackNumber() - query.getCallbackUseNumber());
            result.setEmailLastNumber(query.getEmailNumber() - query.getEmailUseNumber());
            result.setSmsLastNumber(query.getSmsNumber() - query.getSmsUseNumber());
            result.setVoiceLastNumber(query.getVoiceNumber() - query.getVoiceUseNumber());
            response.setData(result);
            response.setRetCode(PlatformCodeEnum.EXECUTE_SUCCESS.getCode());
        } catch (BizException bizException) {
            response.setRetCode(bizException.getCode());
            response.setRetMsg(bizException.getMessage());
            response.setFlag(false);
        } catch (Throwable e) {
            log.error("系统异常", e);
            response.setRetCode(PlatformCodeEnum.SYS_ERROR.getCode());
            response.setRetMsg(PlatformCodeEnum.SYS_ERROR.getDesc());
            response.setFlag(false);
        }
        return response;
    }

    /**
     * 删除监控
     *
     * @param delete
     * @param user
     * @return com.ifourthwall.common.base.BaseResponse
     * @throws
     * @Author: YangPeng
     * @Date: 2020/4/1 14:55
     */
    @Override
    public BaseResponse deleteMonitor(DeleteMonitorDTO delete, IFWUser user) {
        String tenantId = CheckAccessUtils.judgeTenantId(delete.getTenantId(), user);
        delete.setTenantId(tenantId);
        delete.setUpdateBy(user.getUserId());
        return monitorFacade.deleteMonitor(delete);
    }

    /**
     * 查询监控对象标签
     *
     * @param queryList
     * @param user
     * @return com.ifourthwall.common.base.BaseResponse<com.ifourthwall.common.base.IFWPageInfo < com.ifourthwall.dbm.asset.dto.QueryMonitorTagListDTO>>
     * @throws
     * @Author: YangPeng
     * @Date: 2020/4/1 16:41
     */
    @Override
    public BaseResponse<IFWPageInfo<QueryMonitorTagListDTO>> queryTagList(QueryMonitorTagListQuDTO queryList, IFWUser user) {
        String tenantId = CheckAccessUtils.judgeTenantId(queryList.getTenantId(), user);
        queryList.setTenantId(tenantId);
        return monitorFacade.queryTagList(queryList);
    }

    /**
     * sentry实时监控图-告警次数排行
     *
     * @param rankingQuDTO
     * @param user
     * @return
     * @throws
     * @Author: YangPeng
     * @Date: 2020/4/3 11:34
     */
    @Override
    public BaseResponse<alertNumberRanking> alertNumberRanking(SentryMonitorRankingQuDTO rankingQuDTO, IFWUser user) {
        BaseResponse<alertNumberRanking> response = new BaseResponse<>();
        alertNumberRanking result = new alertNumberRanking();
        ArrayList<alertNumberDTO> resultList = new ArrayList<>();
        try {
            String key = null;
            //查询redis中是否存在该数据
            if (!StringUtils.isEmpty(rankingQuDTO.getSpaceId())) {
                //seer端通过空间筛选（项目id + 查询周期 + 空间id）
                key = RedisKeyPrefixUtil.getAssetCountInfoType(rankingQuDTO.getProjectId() + rankingQuDTO.getTime() + rankingQuDTO.getSpaceId() + rankingQuDTO.getLanguageCode());
            }
            if (StringUtils.isEmpty(rankingQuDTO.getSpaceId())) {
                //web端普通筛选
                key = RedisKeyPrefixUtil.getAssetCountInfoType(rankingQuDTO.getProjectId() + rankingQuDTO.getTime() + rankingQuDTO.getLanguageCode());
            }
            //查询redis中是否存在该数据
            alertNumberRanking queryRedis = (alertNumberRanking) redisTemplate.opsForValue().get(RedisKeyPrefixUtil.getSentrySeerAlertNumberRanking(key));
            if (queryRedis == null) {
                Date startTime = null;
                Date endTime = null;
                SimpleDateFormat formatter = new SimpleDateFormat("yyyy-MM-dd HH:mm:ss");
                if ("1".equals(rankingQuDTO.getTime())) {
                    startTime = DataUtils.getDayBegin();
                    endTime = DataUtils.getDayEnd();
                }
                if ("2".equals(rankingQuDTO.getTime())) {
                    startTime = DataUtils.getBeginDayOfWeek();
                    endTime = DataUtils.getEndDayOfWeek();
                }
                if ("3".equals(rankingQuDTO.getTime())) {
                    startTime = DataUtils.getBeginDayOfMonth();
                    endTime = DataUtils.getEndDayOfMonth();
                }
                String tenantId = CheckAccessUtils.judgeTenantId(rankingQuDTO.getTenantId(), user);
                rankingQuDTO.setTenantId(tenantId);
                //如果过是需要通过空间id查询
                ArrayList<String> spaceIds = new ArrayList<>();
                if (!StringUtils.isEmpty(rankingQuDTO.getSpaceId())) {
                    GetSonSpaceQueryBO querySpace = new GetSonSpaceQueryBO();
                    querySpace.setSpaceId(rankingQuDTO.getSpaceId());
                    GetSonSpaceBO sonpace = spaceRepository.getSonpace(querySpace);
                    spaceIds.addAll(sonpace.getSpaceId());
                }
                //查询系统分类
                TranslationDoBO translationDoBO = new TranslationDoBO();
                translationDoBO.setLanguageCode(rankingQuDTO.getLanguageCode());
                AllTranslationDoBO allTranslationDoBO = assetRepository.allTranslation(translationDoBO);
                for (Map.Entry<String, String> tagIds : allTranslationDoBO.getSystem().entrySet()) {
                    alertNumberDTO alertNumberRanking = new alertNumberDTO();
                    alertNumberRanking.setTagName(tagIds.getValue());
                    alertNumberRanking.setTagId(tagIds.getKey());
                    resultList.add(alertNumberRanking);
                }
                //监控所有记录
                QueryRecordingQueryDoBO queryRecordingList = new QueryRecordingQueryDoBO();
                queryRecordingList.setProjectId(rankingQuDTO.getProjectId());
                queryRecordingList.setSpaceId(spaceIds);
                queryRecordingList.setBeginTime(formatter.format(startTime));
                queryRecordingList.setEndTime(formatter.format(endTime));
                queryRecordingList.setLanguageCode(rankingQuDTO.getLanguageCode());
                queryRecordingList.setTenantId(rankingQuDTO.getTenantId());
                List<QueryRecordingListDoBO> recordingAllList = monitorRepository.queryRecordingList(queryRecordingList).getResult();
                for (alertNumberDTO tagId : resultList) {
                    //当没有任何告警记录
                    if (DataUtils.isListAvali(recordingAllList)) {
                        //筛选出有系统的检测对象
                        List<QueryRecordingListDoBO> collect = recordingAllList.stream().filter(a -> a.getAssetSystemId() != null).collect(Collectors.toList());
                        result.setTotal(collect.size());
                        queryRecordingList.setAssetSystemId(tagId.getTagId());
                        //按照资产系统查询监控记录
                        List<QueryRecordingListDoBO> recordingList = monitorRepository.queryRecordingList(queryRecordingList).getResult();
                        if (DataUtils.isListAvali(recordingList)) {
                            tagId.setAlertNumber(recordingList.size());
                        } else {
                            tagId.setAlertNumber(0);
                        }
                    } else {
                        tagId.setAlertNumber(0);
                        result.setTotal(0);
                    }
                }
            }
            //排序
            if (DataUtils.isListAvali(resultList)) {
                List<alertNumberDTO> collect = resultList.stream().sorted(Comparator.comparing(alertNumberDTO::getAlertNumber).reversed()).collect(Collectors.toList());
                result.setRankInfo(collect);
            } else {
                result.setRankInfo(resultList);
            }
            response.setData(result);
//            redisTemplate.opsForValue().set(RedisKeyPrefixUtil.getSentrySeerAlertNumberRanking(key), result, 180, TimeUnit.MINUTES);
//            if (queryRedis != null) {
//                response.setData(queryRedis);
//            }

            response.setRetCode(PlatformCodeEnum.EXECUTE_SUCCESS.getCode());
        } catch (BizException bizException) {
            response.setRetCode(bizException.getCode());
            response.setRetMsg(bizException.getMessage());
            response.setFlag(false);
        } catch (Throwable e) {
            log.error("系统异常", e);
            response.setRetCode(PlatformCodeEnum.SYS_ERROR.getCode());
            response.setRetMsg(PlatformCodeEnum.SYS_ERROR.getDesc());
            response.setFlag(false);
        }
        return response;
    }

    /**
     * sentry实时监控图首页-监控设备统计
     *
     * @param query
     * @param user
     * @return
     * @throws
     * @Author: YangPeng
     * @Date: 2020/4/3 15:18
     */
    @Override
    public BaseResponse<monitorSeerInfoDTO> monitorSeerInfo(monitorSeerInfoQuDTO query, IFWUser user) {
        BaseResponse<monitorSeerInfoDTO> response = new BaseResponse<>();
        monitorSeerInfoDTO result = new monitorSeerInfoDTO();
        try {
            String tenantId = CheckAccessUtils.judgeTenantId(query.getTenantId(), user);
            query.setTenantId(tenantId);
            QueryMonitorListDoQueryBO queryMonitorList = new QueryMonitorListDoQueryBO();
            queryMonitorList.setProjectId(query.getProjectId());
            queryMonitorList.setLanguageCode(query.getLanguageCode());
            queryMonitorList.setTenantId(query.getTenantId());
            List<QueryMonitorListDoBO> monitorList = monitorRepository.queryMonitorList(queryMonitorList).getList();
            if (monitorList != null) {
                result.setTotal(monitorList.size());
                Map<String, List<QueryMonitorListDoBO>> status = monitorList.stream().filter(a -> a.getAssetStatus() != null).collect(Collectors.groupingBy(QueryMonitorListDoBO::getAssetStatus));
                for (Map.Entry<String, List<QueryMonitorListDoBO>> statusId : status.entrySet()) {
                    if ("2".equals(statusId.getKey())) {
                        result.setFaultAsset(statusId.getValue().size());
                    } else {
                        result.setNormalAsset(statusId.getValue().size());
                    }
                }
            }
            response.setData(result);
            response.setRetCode(PlatformCodeEnum.EXECUTE_SUCCESS.getCode());
        } catch (BizException bizException) {
            response.setRetCode(bizException.getCode());
            response.setRetMsg(bizException.getMessage());
            response.setFlag(false);
        } catch (Throwable e) {
            log.error("系统异常", e);
            response.setRetCode(PlatformCodeEnum.SYS_ERROR.getCode());
            response.setRetMsg(PlatformCodeEnum.SYS_ERROR.getDesc());
            response.setFlag(false);
        }
        return response;
    }

    /**
     * seer查询sentry监控
     *
     * @param query
     * @param user
     * @return
     * @throws
     * @Author: YangPeng
     * @Date: 2020/4/16 17:57
     */
    @Override
    public BaseResponse<SeerQuerySentryInfoDTO> seerQuerySentryInfo(SeerQuerySentryInfoQuDTO query, IFWUser user) {
        BaseResponse<SeerQuerySentryInfoDTO> response = new BaseResponse<>();
        SeerQuerySentryInfoDTO result = new SeerQuerySentryInfoDTO();
        try {
            String tenantId = CheckAccessUtils.judgeTenantId(query.getTenantId(), user);
            query.setTenantId(tenantId);
            QueryMonitorListDoQueryBO queryMonitorList = new QueryMonitorListDoQueryBO();
            //查询空间下的子空间
            ArrayList<String> spaceIds = new ArrayList<>();
            if (StringUtils.isEmpty(query.getSpaceId())) {
                GetSonSpaceQueryBO querySapceId = new GetSonSpaceQueryBO();
                querySapceId.setSpaceId(query.getSpaceId());
                GetSonSpaceBO sonpace = spaceRepository.getSonpace(querySapceId);
                if (DataUtils.isListAvali(sonpace.getSpaceId())) {
                    spaceIds.addAll(sonpace.getSpaceId());
                }
                queryMonitorList.setSpaceId(spaceIds);
            }
            queryMonitorList.setProjectId(query.getProjectId());
            queryMonitorList.setAssetStatus("2");
            IFWPageInfo<QueryMonitorListDoBO> monitorList = monitorRepository.queryMonitorList(queryMonitorList);
            if (monitorList != null) {
                result.setAlertNumber(monitorList.getList().size());
            }
            result.setSpaceId(query.getSpaceId());
            response.setData(result);
            response.setRetCode(PlatformCodeEnum.EXECUTE_SUCCESS.getCode());
        } catch (BizException bizException) {
            response.setRetCode(bizException.getCode());
            response.setRetMsg(bizException.getMessage());
            response.setFlag(false);
        } catch (Throwable e) {
            log.error("系统异常", e);
            response.setRetCode(PlatformCodeEnum.SYS_ERROR.getCode());
            response.setRetMsg(PlatformCodeEnum.SYS_ERROR.getDesc());
            response.setFlag(false);
        }
        return response;
    }

    /**
     * 查询是否引用模板
     *
     * @param query
     * @param user
     * @return com.ifourthwall.common.base.BaseResponse<java.lang.Boolean>
     * @throws
     * @Author: YangPeng
     * @Date: 2020/4/17 18:10
     */
    @Override
    public BaseResponse<Boolean> sentryQueryTask(SentryQueryTaskQuDTO query, IFWUser user) {
        String tenantId = CheckAccessUtils.judgeTenantId(query.getTenantId(), user);
        query.setTenantId(tenantId);
        return monitorFacade.sentryQueryTask(query);
    }

    /**
     * 查询用电用水信息
     *
     * @param query
     * @param user
     * @return com.ifourthwall.common.base.BaseResponse<com.ifourthwall.dbm.asset.dto.QueryAllByStatisticsDTO>
     * @throws
     * @Author: YangPeng
     * @Date: 2020/4/22 19:08
     */
    @Override
    public BaseResponse<QueryAllByStatisticsDTO> queryAllByStatistics(QueryAllByStatisticsQuDTO query, IFWUser user) {
        String tenantId = CheckAccessUtils.judgeTenantId(query.getTenantId(), user);
        query.setTenantId(tenantId);
        return monitorFacade.queryAllByStatistics(query);
    }

    @Override
    public BaseResponse<queryDataPointIdDTO> queryDataPointId(queryDataPointIdQuDTO upMonitorRecordDTO) {
        return monitorFacade.queryDataPointId(upMonitorRecordDTO);
    }


    public static void main(String[] args) {
        for (int i = 0; i < 10; i++) {

            System.out.println(IFWUUIDUtils.randomUUID());
        }

    }
}


