/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.hive.llap.cache;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Preconditions;
import java.io.Closeable;
import java.io.File;
import java.io.IOException;
import java.io.RandomAccessFile;
import java.nio.ByteBuffer;
import java.nio.MappedByteBuffer;
import java.nio.channels.FileChannel;
import java.nio.file.FileSystems;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.attribute.FileAttribute;
import java.nio.file.attribute.PosixFilePermission;
import java.nio.file.attribute.PosixFilePermissions;
import java.util.Arrays;
import java.util.Set;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.atomic.AtomicLong;
import java.util.concurrent.locks.ReentrantLock;
import org.apache.commons.io.IOUtils;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.hive.common.io.Allocator;
import org.apache.hadoop.hive.common.io.encoded.MemoryBuffer;
import org.apache.hadoop.hive.conf.HiveConf;
import org.apache.hadoop.hive.llap.cache.BuddyAllocatorMXBean;
import org.apache.hadoop.hive.llap.cache.EvictionAwareAllocator;
import org.apache.hadoop.hive.llap.cache.LlapAllocatorBuffer;
import org.apache.hadoop.hive.llap.cache.LlapDataBuffer;
import org.apache.hadoop.hive.llap.cache.LlapOomDebugDump;
import org.apache.hadoop.hive.llap.cache.MemoryManager;
import org.apache.hadoop.hive.llap.io.api.impl.LlapIoImpl;
import org.apache.hadoop.hive.llap.metrics.LlapDaemonCacheMetrics;

public final class BuddyAllocator
implements EvictionAwareAllocator,
BuddyAllocatorMXBean,
LlapOomDebugDump {
    private final Arena[] arenas;
    private final AtomicInteger allocatedArenas = new AtomicInteger(0);
    private final MemoryManager memoryManager;
    private static final long MAX_DUMP_INTERVAL_NS = 300000000000L;
    private final AtomicLong lastLog = new AtomicLong(-1L);
    private final LlapDaemonCacheMetrics metrics;
    private static final int MAX_DISCARD_ATTEMPTS = 10;
    private static final int LOG_DISCARD_ATTEMPTS = 5;
    private final int minAllocLog2;
    private final int maxAllocLog2;
    private final int arenaSizeLog2;
    private final int maxArenas;
    private final int minAllocation;
    private final int maxAllocation;
    private final int arenaSize;
    private final long maxSize;
    private final boolean isDirect;
    private final boolean isMapped;
    private final Path cacheDir;
    private boolean enableDefragShortcut = true;
    private boolean oomLogging = true;
    private static final int MAX_ARENA_SIZE = 0x40000000;
    private static final int MIN_TOTAL_MEMORY_SIZE = 0x4000000;
    private static final float MAX_DEFRAG_HEADROOM_FRACTION = 0.01f;
    private static final FileAttribute<Set<PosixFilePermission>> RWX = PosixFilePermissions.asFileAttribute(PosixFilePermissions.fromString("rwx------"));
    private final AtomicLong[] defragCounters;
    private final boolean doUseFreeListDiscard;
    private final boolean doUseBruteDiscard;
    private static final boolean assertsEnabled = BuddyAllocator.areAssertsEnabled();
    private static final ThreadLocal<DiscardContext> threadCtx = new ThreadLocal<DiscardContext>(){

        @Override
        protected DiscardContext initialValue() {
            return new DiscardContext();
        }
    };

    public BuddyAllocator(Configuration conf, MemoryManager mm, LlapDaemonCacheMetrics metrics) {
        this(HiveConf.getBoolVar((Configuration)conf, (HiveConf.ConfVars)HiveConf.ConfVars.LLAP_ALLOCATOR_DIRECT), HiveConf.getBoolVar((Configuration)conf, (HiveConf.ConfVars)HiveConf.ConfVars.LLAP_ALLOCATOR_MAPPED), (int)HiveConf.getSizeVar((Configuration)conf, (HiveConf.ConfVars)HiveConf.ConfVars.LLAP_ALLOCATOR_MIN_ALLOC), (int)HiveConf.getSizeVar((Configuration)conf, (HiveConf.ConfVars)HiveConf.ConfVars.LLAP_ALLOCATOR_MAX_ALLOC), HiveConf.getIntVar((Configuration)conf, (HiveConf.ConfVars)HiveConf.ConfVars.LLAP_ALLOCATOR_ARENA_COUNT), BuddyAllocator.getMaxTotalMemorySize(conf), HiveConf.getSizeVar((Configuration)conf, (HiveConf.ConfVars)HiveConf.ConfVars.LLAP_ALLOCATOR_DEFRAG_HEADROOM), HiveConf.getVar((Configuration)conf, (HiveConf.ConfVars)HiveConf.ConfVars.LLAP_ALLOCATOR_MAPPED_PATH), mm, metrics, HiveConf.getVar((Configuration)conf, (HiveConf.ConfVars)HiveConf.ConfVars.LLAP_ALLOCATOR_DISCARD_METHOD));
    }

    private static boolean areAssertsEnabled() {
        boolean assertsEnabled = false;
        if (!$assertionsDisabled) {
            assertsEnabled = true;
            if (!true) {
                throw new AssertionError();
            }
        }
        return assertsEnabled;
    }

    private static long getMaxTotalMemorySize(Configuration conf) {
        long maxSize = HiveConf.getSizeVar((Configuration)conf, (HiveConf.ConfVars)HiveConf.ConfVars.LLAP_IO_MEMORY_MAX_SIZE);
        if (maxSize > 0x4000000L || HiveConf.getBoolVar((Configuration)conf, (HiveConf.ConfVars)HiveConf.ConfVars.HIVE_IN_TEST)) {
            return maxSize;
        }
        throw new RuntimeException("Allocator space is too small for reasonable operation; " + HiveConf.ConfVars.LLAP_IO_MEMORY_MAX_SIZE.varname + "=" + maxSize + ", but at least " + 0x4000000 + " is required. If you cannot spare any memory, you can disable LLAP IO entirely via " + HiveConf.ConfVars.LLAP_IO_ENABLED.varname);
    }

    @VisibleForTesting
    public BuddyAllocator(boolean isDirectVal, boolean isMappedVal, int minAllocVal, int maxAllocVal, int arenaCount, long maxSizeVal, long defragHeadroom, String mapPath, MemoryManager memoryManager, LlapDaemonCacheMetrics metrics, String discardMethod) {
        this.isDirect = isDirectVal;
        this.isMapped = isMappedVal;
        this.minAllocation = minAllocVal;
        this.maxAllocation = maxAllocVal;
        if (this.isMapped) {
            try {
                this.cacheDir = Files.createTempDirectory(FileSystems.getDefault().getPath(mapPath, new String[0]), "llap-", RWX);
            }
            catch (IOException ioe) {
                throw new AssertionError("Configured mmap directory should be writable", ioe);
            }
        } else {
            this.cacheDir = null;
        }
        this.arenaSize = this.validateAndDetermineArenaSize(arenaCount, maxSizeVal);
        this.maxSize = this.validateAndDetermineMaxSize(maxSizeVal);
        memoryManager.updateMaxSize(this.determineMaxMmSize(defragHeadroom, this.maxSize));
        this.minAllocLog2 = 31 - Integer.numberOfLeadingZeros(this.minAllocation);
        this.maxAllocLog2 = 31 - Integer.numberOfLeadingZeros(this.maxAllocation);
        this.arenaSizeLog2 = 63 - Long.numberOfLeadingZeros(this.arenaSize);
        this.maxArenas = (int)(this.maxSize / (long)this.arenaSize);
        this.arenas = new Arena[this.maxArenas];
        for (int i = 0; i < this.maxArenas; ++i) {
            this.arenas[i] = new Arena();
        }
        Arena firstArena = this.arenas[0];
        firstArena.init(0);
        this.allocatedArenas.set(1);
        this.memoryManager = memoryManager;
        this.defragCounters = new AtomicLong[this.maxAllocLog2 - this.minAllocLog2 + 1];
        for (int i = 0; i < this.defragCounters.length; ++i) {
            this.defragCounters[i] = new AtomicLong(0L);
        }
        this.metrics = metrics;
        metrics.incrAllocatedArena();
        boolean isBoth = null == discardMethod || "both".equalsIgnoreCase(discardMethod);
        this.doUseFreeListDiscard = isBoth || "freelist".equalsIgnoreCase(discardMethod);
        this.doUseBruteDiscard = isBoth || "brute".equalsIgnoreCase(discardMethod);
    }

    public long determineMaxMmSize(long defragHeadroom, long maxMmSize) {
        if (defragHeadroom > 0L) {
            long maxHeadroom = (long)Math.floor((float)this.maxSize * 0.01f);
            defragHeadroom = Math.min(maxHeadroom, defragHeadroom);
            LlapIoImpl.LOG.info("Leaving " + defragHeadroom + " of defragmentation headroom");
            maxMmSize -= defragHeadroom;
        }
        return maxMmSize;
    }

    public long validateAndDetermineMaxSize(long maxSizeVal) {
        if (maxSizeVal % (long)this.arenaSize > 0L) {
            long oldMaxSize = maxSizeVal;
            maxSizeVal = maxSizeVal / (long)this.arenaSize * (long)this.arenaSize;
            LlapIoImpl.LOG.warn("Rounding cache size to " + maxSizeVal + " from " + oldMaxSize + " to be divisible by arena size " + this.arenaSize);
        }
        if (maxSizeVal / (long)this.arenaSize > Integer.MAX_VALUE) {
            throw new RuntimeException("Too many arenas needed to allocate the cache: " + this.arenaSize + ", " + maxSizeVal);
        }
        return maxSizeVal;
    }

    public int validateAndDetermineArenaSize(int arenaCount, long maxSizeVal) {
        long arenaSizeVal = arenaCount == 0 ? 0x40000000L : maxSizeVal / (long)arenaCount;
        arenaSizeVal = Math.max((long)this.maxAllocation, Math.min(arenaSizeVal, 0x40000000L));
        if (LlapIoImpl.LOG.isInfoEnabled()) {
            LlapIoImpl.LOG.info("Buddy allocator with " + (this.isDirect ? "direct" : "byte") + " buffers; " + (this.isMapped ? "memory mapped off " + this.cacheDir.toString() + "; " : "") + "allocation sizes " + this.minAllocation + " - " + this.maxAllocation + ", arena size " + arenaSizeVal + ", total size " + maxSizeVal);
        }
        String minName = HiveConf.ConfVars.LLAP_ALLOCATOR_MIN_ALLOC.varname;
        String maxName = HiveConf.ConfVars.LLAP_ALLOCATOR_MAX_ALLOC.varname;
        if (this.minAllocation < 8) {
            throw new RuntimeException(minName + " must be at least 8 bytes: " + this.minAllocation);
        }
        if (maxSizeVal < (long)this.maxAllocation || this.maxAllocation < this.minAllocation) {
            throw new RuntimeException("Inconsistent sizes; expecting " + minName + " <= " + maxName + " <= " + HiveConf.ConfVars.LLAP_IO_MEMORY_MAX_SIZE.varname + "; configured with min=" + this.minAllocation + ", max=" + this.maxAllocation + " and total=" + maxSizeVal);
        }
        if (Integer.bitCount(this.minAllocation) != 1 || Integer.bitCount(this.maxAllocation) != 1) {
            throw new RuntimeException("Allocation sizes must be powers of two; configured with " + minName + "=" + this.minAllocation + ", " + maxName + "=" + this.maxAllocation);
        }
        if (arenaSizeVal % (long)this.maxAllocation > 0L) {
            long oldArenaSize = arenaSizeVal;
            arenaSizeVal = arenaSizeVal / (long)this.maxAllocation * (long)this.maxAllocation;
            LlapIoImpl.LOG.warn("Rounding arena size to " + arenaSizeVal + " from " + oldArenaSize + " to be divisible by allocation size " + this.maxAllocation);
        }
        return (int)arenaSizeVal;
    }

    public void allocateMultiple(MemoryBuffer[] dest, int size) throws Allocator.AllocatorOutOfMemoryException {
        this.allocateMultiple(dest, size, null);
    }

    public void allocateMultiple(MemoryBuffer[] dest, int size, Allocator.BufferObjectFactory factory) throws Allocator.AllocatorOutOfMemoryException {
        long threadId;
        int destAllocIx;
        assert (size > 0) : "size is " + size;
        if (size > this.maxAllocation) {
            throw new RuntimeException("Trying to allocate " + size + "; max is " + this.maxAllocation);
        }
        int freeListIx = this.determineFreeListForAllocation(size);
        int allocLog2 = freeListIx + this.minAllocLog2;
        int allocationSize = 1 << allocLog2;
        this.memoryManager.reserveMemory(dest.length << allocLog2);
        for (int i = 0; i < dest.length; ++i) {
            if (dest[i] != null) continue;
            dest[i] = factory != null ? factory.create() : this.createUnallocated();
        }
        int arenaCount = this.allocatedArenas.get();
        if (arenaCount < 0) {
            arenaCount = -arenaCount - 1;
        }
        if ((destAllocIx = this.allocateFast(dest, null, 0, dest.length, freeListIx, allocationSize, (int)((threadId = arenaCount > 1 ? Thread.currentThread().getId() : 0L) % (long)arenaCount), arenaCount)) == dest.length) {
            return;
        }
        int attempt = 0;
        boolean isFailed = false;
        int memoryForceReleased = 0;
        try {
            int discardFailed = 0;
            while (true) {
                int startArenaIx;
                if ((destAllocIx = this.allocateWithSplit(dest, null, destAllocIx, dest.length, freeListIx, allocationSize, startArenaIx = (int)((threadId + (long)attempt) % (long)arenaCount), arenaCount, -1)) == dest.length) {
                    return;
                }
                if (attempt == 0 && (destAllocIx = this.allocateWithExpand(dest, destAllocIx, freeListIx, allocationSize, arenaCount)) == dest.length) {
                    return;
                }
                boolean hasDiscardedAny = false;
                DiscardContext ctx = threadCtx.get();
                int maxListSize = 1 << (this.doUseBruteDiscard ? freeListIx : freeListIx - 1);
                int requiredBlocks = dest.length - destAllocIx;
                ctx.init(maxListSize, requiredBlocks);
                if (this.doUseFreeListDiscard && freeListIx > 0) {
                    this.discardBlocksBasedOnFreeLists(freeListIx, startArenaIx, arenaCount, ctx);
                    memoryForceReleased += ctx.memoryReleased;
                    hasDiscardedAny = ctx.resultCount > 0;
                    destAllocIx = this.allocateFromDiscardResult(dest, destAllocIx, freeListIx, allocationSize, ctx);
                    if (destAllocIx == dest.length) {
                        return;
                    }
                }
                if (this.doUseBruteDiscard) {
                    ctx.resetResults();
                    this.discardBlocksBruteForce(freeListIx, startArenaIx, arenaCount, ctx);
                    memoryForceReleased += ctx.memoryReleased;
                    hasDiscardedAny = hasDiscardedAny || ctx.resultCount > 0;
                    destAllocIx = this.allocateFromDiscardResult(dest, destAllocIx, freeListIx, allocationSize, ctx);
                    if (destAllocIx == dest.length) {
                        return;
                    }
                }
                if (hasDiscardedAny) {
                    discardFailed = 0;
                } else if (++discardFailed > 10) {
                    String msg = "Failed to allocate " + size + "; at " + destAllocIx + " out of " + dest.length + " (entire cache is fragmented and locked, or an internal issue)";
                    this.logOomErrorMessage(msg);
                    isFailed = true;
                    throw new Allocator.AllocatorOutOfMemoryException(msg);
                }
                ++attempt;
            }
        }
        finally {
            this.memoryManager.releaseMemory(memoryForceReleased);
            if (!isFailed && attempt >= 5) {
                LlapIoImpl.LOG.info("Allocation of " + dest.length + " buffers of size " + size + " took " + attempt + " attempts to free enough memory; force-released " + memoryForceReleased);
            }
        }
    }

    private void discardBlocksBasedOnFreeLists(int freeListIx, int startArenaIx, int arenaCount, DiscardContext ctx) {
        this.defragCounters[freeListIx].incrementAndGet();
        int mergeListIx = freeListIx - 1;
        int arenaIx = startArenaIx;
        do {
            Arena arena = this.arenas[arenaIx];
            arena.reserveDiscardBlocksBasedOnFreeList(mergeListIx, ctx);
            this.discardFromCtxBasedOnFreeList(arena, ctx, freeListIx);
            if (ctx.remainingToFind == 0) {
                return;
            }
            ctx.resetBetweenArenas();
        } while ((arenaIx = BuddyAllocator.getNextIx(arenaIx, arenaCount, 1)) != startArenaIx);
    }

    private void discardBlocksBruteForce(int freeListIx, int startArenaIx, int arenaCount, DiscardContext ctx) {
        long counter = this.defragCounters[freeListIx].incrementAndGet();
        int positionsPerArena = 1 << this.arenaSizeLog2 - (this.minAllocLog2 + freeListIx);
        int startHeaderIx = (int)(counter % (long)positionsPerArena) << freeListIx;
        int arenaIx = startArenaIx;
        do {
            Arena arena = this.arenas[arenaIx];
            arena.reserveDiscardBruteForce(freeListIx, ctx, startHeaderIx);
            this.discardFromCtxBruteForce(arena, ctx, freeListIx);
            if (ctx.remainingToFind == 0) {
                return;
            }
            ctx.resetBetweenArenas();
        } while ((arenaIx = BuddyAllocator.getNextIx(arenaIx, arenaCount, 1)) != startArenaIx);
    }

    private void discardFromCtxBasedOnFreeList(Arena arena, DiscardContext ctx, int freeListIx) {
        this.discardAllBuffersFromCtx(arena, ctx);
        for (int baseIx = ctx.baseCount - 1; baseIx >= 0; --baseIx) {
            int baseHeaderIx = ctx.baseHeaders[baseIx];
            int minHeaderIx = Math.min(baseHeaderIx, BuddyAllocator.getBuddyHeaderIx(freeListIx - 1, baseHeaderIx));
            this.finalizeDiscardResult(arena, ctx, freeListIx, minHeaderIx);
        }
    }

    private void discardFromCtxBruteForce(Arena arena, DiscardContext ctx, int freeListIx) {
        this.discardAllBuffersFromCtx(arena, ctx);
        for (int baseIx = ctx.baseCount - 1; baseIx >= 0; --baseIx) {
            this.finalizeDiscardResult(arena, ctx, freeListIx, ctx.baseHeaders[baseIx]);
        }
    }

    private void finalizeDiscardResult(Arena arena, DiscardContext ctx, int freeListIx, int newlyFreeHeaderIx) {
        int maxHeaderIx = newlyFreeHeaderIx + (1 << freeListIx);
        if (assertsEnabled) {
            arena.checkHeader(newlyFreeHeaderIx, -1, true);
        }
        arena.unsetHeaders(newlyFreeHeaderIx + 1, maxHeaderIx, CasLog.Src.CLEARED_VICTIM);
        arena.setHeaderNoBufAlloc(newlyFreeHeaderIx, freeListIx, CasLog.Src.NEWLY_CLEARED);
        ctx.addResult(arena.arenaIx, newlyFreeHeaderIx);
    }

    private void discardAllBuffersFromCtx(Arena arena, DiscardContext ctx) {
        for (int victimIx = 0; victimIx < ctx.victimCount; ++victimIx) {
            int victimHeaderIx = ctx.victimHeaders[victimIx];
            LlapAllocatorBuffer buf = arena.buffers[victimHeaderIx];
            if (buf == null) continue;
            if (assertsEnabled) {
                arena.checkHeader(victimHeaderIx, -1, true);
                byte header = arena.headers[victimHeaderIx];
                this.assertBufferLooksValid(BuddyAllocator.freeListFromHeader(header), buf, arena.arenaIx, victimHeaderIx);
            }
            ((Arena)arena).buffers[victimHeaderIx] = null;
            long memUsage = buf.getMemoryUsage();
            Boolean result = buf.endDiscard();
            if (result == null) {
                ctx.memoryReleased = (int)((long)ctx.memoryReleased + memUsage);
                continue;
            }
            if (!result.booleanValue()) continue;
            this.memoryManager.releaseMemory(memUsage);
        }
    }

    private void cancelDiscard(LlapAllocatorBuffer buf, int arenaIx, int headerIx) {
        Boolean result = buf.cancelDiscard();
        if (result == null) {
            return;
        }
        if (result.booleanValue()) {
            long memUsage = buf.getMemoryUsage();
            this.arenas[arenaIx].deallocate(buf, true);
            this.memoryManager.releaseMemory(memUsage);
        } else {
            this.arenas[arenaIx].deallocate(buf, true);
        }
    }

    private int allocateFast(MemoryBuffer[] dest, long[] destHeaders, int destIx, int destCount, int freeListIx, int allocSize, int startArenaIx, int arenaCount) {
        int index = startArenaIx;
        do {
            int newDestIx;
            if ((newDestIx = this.arenas[index].allocateFast(freeListIx, dest, destHeaders, destIx, destCount, allocSize)) == destCount) {
                return newDestIx;
            }
            assert (newDestIx != -1);
            destIx = newDestIx;
        } while ((index = BuddyAllocator.getNextIx(index, arenaCount, 1)) != startArenaIx);
        return destIx;
    }

    private int allocateWithExpand(MemoryBuffer[] dest, int destIx, int freeListIx, int allocSize, int arenaCount) {
        for (int arenaIx = arenaCount; arenaIx < this.arenas.length; ++arenaIx) {
            if ((destIx = this.arenas[arenaIx].allocateWithExpand(arenaIx, freeListIx, dest, destIx, allocSize)) != dest.length) continue;
            return destIx;
        }
        return destIx;
    }

    private int allocateWithSplit(MemoryBuffer[] dest, long[] destHeaders, int destIx, int destCount, int freeListIx, int allocSize, int startArenaIx, int arenaCount, int maxSplitFreeListIx) {
        int arenaIx = startArenaIx;
        do {
            int newDestIx;
            if ((newDestIx = this.arenas[arenaIx].allocateWithSplit(freeListIx, dest, destHeaders, destIx, destCount, allocSize, maxSplitFreeListIx)) == destCount) {
                return newDestIx;
            }
            assert (newDestIx != -1);
            destIx = newDestIx;
        } while ((arenaIx = BuddyAllocator.getNextIx(arenaIx, arenaCount, 1)) != startArenaIx);
        return destIx;
    }

    private int allocateFromDiscardResult(MemoryBuffer[] dest, int destAllocIx, int freeListIx, int allocationSize, DiscardContext discardResult) {
        for (int i = 0; i < discardResult.resultCount; ++i) {
            long result = discardResult.results[i];
            destAllocIx = this.arenas[BuddyAllocator.getFirstInt(result)].allocateFromDiscard(dest, destAllocIx, BuddyAllocator.getSecondInt(result), freeListIx, allocationSize);
        }
        return destAllocIx;
    }

    private void logOomErrorMessage(String msg) {
        long time;
        long lastTime;
        boolean shouldLog;
        if (!this.oomLogging) {
            return;
        }
        do {
            time = System.nanoTime();
            lastTime = this.lastLog.get();
            boolean bl = shouldLog = lastTime == -1L || time - lastTime > 300000000000L;
        } while (shouldLog && !this.lastLog.compareAndSet(lastTime, time));
        if (shouldLog) {
            LlapIoImpl.LOG.error(msg + this.debugDumpForOomInternal());
        } else {
            LlapIoImpl.LOG.error(msg);
        }
    }

    @Override
    public void debugDumpShort(StringBuilder sb) {
        this.memoryManager.debugDumpShort(sb);
        sb.append("\nDefrag counters: ");
        for (int i = 0; i < this.defragCounters.length; ++i) {
            sb.append(this.defragCounters[i].get()).append(", ");
        }
        sb.append("\nAllocator state:");
        int unallocCount = 0;
        int fullCount = 0;
        long totalFree = 0L;
        for (Arena arena : this.arenas) {
            Integer result = arena.debugDumpShort(sb);
            if (result == null) {
                ++unallocCount;
                continue;
            }
            if (result == 0) {
                ++fullCount;
                continue;
            }
            totalFree += (long)result.intValue();
        }
        sb.append("\nTotal available and allocated: ").append(totalFree).append("; unallocated arenas: ").append(unallocCount).append("; full arenas ").append(fullCount);
        sb.append("\n");
    }

    public void deallocate(MemoryBuffer buffer) {
        LlapAllocatorBuffer buf = (LlapAllocatorBuffer)buffer;
        int arenaToRelease = buf.invalidateAndRelease();
        if (arenaToRelease < 0) {
            return;
        }
        long memUsage = buf.getMemoryUsage();
        this.arenas[arenaToRelease].deallocate(buf, false);
        this.memoryManager.releaseMemory(memUsage);
    }

    @Override
    public void deallocateEvicted(MemoryBuffer buffer) {
        LlapAllocatorBuffer buf = (LlapAllocatorBuffer)buffer;
        assert (buf.isInvalid());
        int arenaToRelease = buf.releaseInvalidated();
        if (arenaToRelease < 0) {
            return;
        }
        this.arenas[arenaToRelease].deallocate(buf, false);
    }

    public boolean isDirectAlloc() {
        return this.isDirect;
    }

    private ByteBuffer preallocateArenaBuffer(int arenaSize) {
        if (this.isMapped) {
            MappedByteBuffer mappedByteBuffer;
            RandomAccessFile rwf = null;
            File rf = null;
            Preconditions.checkArgument((boolean)this.isDirect, (Object)"All memory mapped allocations have to be direct buffers");
            try {
                MappedByteBuffer rwbuf;
                rf = File.createTempFile("arena-", ".cache", this.cacheDir.toFile());
                rwf = new RandomAccessFile(rf, "rw");
                rwf.setLength(arenaSize);
                mappedByteBuffer = rwbuf = rwf.getChannel().map(FileChannel.MapMode.READ_WRITE, 0L, arenaSize);
            }
            catch (IOException ioe) {
                try {
                    LlapIoImpl.LOG.warn("Failed trying to allocate memory mapped arena", (Throwable)ioe);
                    throw new OutOfMemoryError("Failed trying to allocate memory mapped arena: " + ioe.getMessage());
                }
                catch (Throwable throwable) {
                    IOUtils.closeQuietly(rwf);
                    if (rf != null) {
                        rf.delete();
                    }
                    throw throwable;
                }
            }
            IOUtils.closeQuietly((Closeable)rwf);
            if (rf != null) {
                rf.delete();
            }
            return mappedByteBuffer;
        }
        return this.isDirect ? ByteBuffer.allocateDirect(arenaSize) : ByteBuffer.allocate(arenaSize);
    }

    @Deprecated
    public MemoryBuffer createUnallocated() {
        return new LlapDataBuffer();
    }

    @Override
    public boolean getIsDirect() {
        return this.isDirect;
    }

    @Override
    public int getMinAllocation() {
        return this.minAllocation;
    }

    @Override
    public int getMaxAllocation() {
        return this.maxAllocation;
    }

    @Override
    public int getArenaSize() {
        return this.arenaSize;
    }

    @Override
    public long getMaxCacheSize() {
        return this.maxSize;
    }

    private static int getBuddyHeaderIx(int freeListIx, int headerIx) {
        return headerIx ^ 1 << freeListIx;
    }

    private static int getNextIx(int ix, int count, int step) {
        assert ((ix += step) <= count);
        return ix == count ? 0 : ix;
    }

    private static int freeListFromHeader(byte header) {
        return (header >> 1) - 1;
    }

    private int freeListFromAllocSize(int allocSize) {
        return 31 - Integer.numberOfLeadingZeros(allocSize) - this.minAllocLog2;
    }

    private int allocSizeFromFreeList(int freeListIx) {
        return 1 << freeListIx + this.minAllocLog2;
    }

    public int offsetFromHeaderIndex(int lastSplitNextHeader) {
        return lastSplitNextHeader << this.minAllocLog2;
    }

    private static byte makeHeader(int freeListIx, boolean isInUse) {
        return (byte)(freeListIx + 1 << 1 | (isInUse ? 1 : 0));
    }

    private int determineFreeListForAllocation(int size) {
        int freeListIx = 31 - Integer.numberOfLeadingZeros(size);
        if (size != 1 << freeListIx) {
            ++freeListIx;
        }
        return Math.max(freeListIx - this.minAllocLog2, 0);
    }

    private static long makeIntPair(int first, int second) {
        return (long)first << 32 | (long)second;
    }

    private static int getFirstInt(long result) {
        return (int)(result >>> 32);
    }

    private static int getSecondInt(long result) {
        return (int)(result & 0xFFFFFFFFL);
    }

    private void assertBufferLooksValid(int freeListIx, LlapAllocatorBuffer buf, int arenaIx, int headerIx) {
        if (buf.allocSize == this.allocSizeFromFreeList(freeListIx)) {
            return;
        }
        this.failWithLog("Race; allocation size " + buf.allocSize + ", not " + this.allocSizeFromFreeList(freeListIx) + " for free list " + freeListIx + " at " + arenaIx + ":" + headerIx);
    }

    private static String toDebugString(LlapAllocatorBuffer buffer) {
        return buffer == null ? "null" : buffer.toDebugString();
    }

    private void checkHeaderByte(int arenaIx, int headerIx, int freeListIx, boolean isLocked, byte header) {
        if (isLocked != ((header & 1) == 1)) {
            this.failWithLog("Expected " + arenaIx + ":" + headerIx + " " + (isLocked ? "" : "not ") + "locked: " + header);
        }
        if (freeListIx < 0) {
            return;
        }
        if (BuddyAllocator.freeListFromHeader(header) != freeListIx) {
            this.failWithLog("Expected " + arenaIx + ":" + headerIx + " in list " + freeListIx + ": " + BuddyAllocator.freeListFromHeader(header));
        }
    }

    @VisibleForTesting
    void disableDefragShortcutForTest() {
        this.enableDefragShortcut = false;
    }

    @VisibleForTesting
    void setOomLoggingForTest(boolean b) {
        this.oomLogging = b;
    }

    @VisibleForTesting
    String testDump() {
        StringBuilder sb = new StringBuilder();
        for (Arena a : this.arenas) {
            a.testDump(sb);
        }
        return sb.toString();
    }

    @Override
    public String debugDumpForOom() {
        return "\nALLOCATOR STATE:\n" + this.debugDumpForOomInternal() + "\nPARENT STATE:\n" + this.memoryManager.debugDumpForOom();
    }

    private String debugDumpForOomInternal() {
        StringBuilder sb = new StringBuilder();
        for (Arena a : this.arenas) {
            a.debugDump(sb);
        }
        return sb.toString();
    }

    private void failWithLog(String string) {
        CasLog.logError();
        throw new AssertionError((Object)string);
    }

    @VisibleForTesting
    public void dumpTestLog() {
        if (CasLog.casLog != null) {
            CasLog.casLog.dumpLog(true);
        }
    }

    private static final class CasLog {
        private static final CasLog casLog = null;
        private final int size;
        private final long[] log;
        private final AtomicInteger offset = new AtomicInteger(0);
        public static final int START_MOVE = 0;
        public static final int SET_NB = 1;
        public static final int SET_BUF = 2;
        public static final int SET_FREE = 3;
        public static final int ADD_TO_LIST = 4;
        public static final int REMOVE_FROM_LIST = 5;
        public static final int ERROR = 6;
        public static final int UNSET = 7;

        public CasLog() {
            this.size = 50000000;
            this.log = new long[this.size];
        }

        public static void logMove(int arenaIx, int buddyHeaderIx, int identityHashCode) {
            if (casLog == null) {
                return;
            }
            int ix = CasLog.casLog.offset.addAndGet(3) - 3;
            CasLog.casLog.log[ix] = BuddyAllocator.makeIntPair(0, identityHashCode);
            CasLog.casLog.log[ix + 1] = Thread.currentThread().getId();
            CasLog.casLog.log[ix + 2] = BuddyAllocator.makeIntPair(arenaIx, buddyHeaderIx);
        }

        public static void logSetNb(Src src, int arenaIx, int headerIndex, int size) {
            if (casLog == null) {
                return;
            }
            int ix = CasLog.casLog.offset.addAndGet(4) - 4;
            CasLog.casLog.log[ix] = BuddyAllocator.makeIntPair(1, src.ordinal());
            CasLog.casLog.log[ix + 1] = Thread.currentThread().getId();
            CasLog.casLog.log[ix + 2] = BuddyAllocator.makeIntPair(arenaIx, headerIndex);
            CasLog.casLog.log[ix + 3] = size;
        }

        public static void logSetFree(Src src, int arenaIx, int headerIndex, int size) {
            if (casLog == null) {
                return;
            }
            int ix = CasLog.casLog.offset.addAndGet(4) - 4;
            CasLog.casLog.log[ix] = BuddyAllocator.makeIntPair(3, src.ordinal());
            CasLog.casLog.log[ix + 1] = Thread.currentThread().getId();
            CasLog.casLog.log[ix + 2] = BuddyAllocator.makeIntPair(arenaIx, headerIndex);
            CasLog.casLog.log[ix + 3] = size;
        }

        public static void logUnset(Src src, int arenaIx, int from, int to) {
            if (casLog == null) {
                return;
            }
            if (from > to) {
                return;
            }
            int ix = CasLog.casLog.offset.addAndGet(4) - 4;
            CasLog.casLog.log[ix] = BuddyAllocator.makeIntPair(7, src.ordinal());
            CasLog.casLog.log[ix + 1] = Thread.currentThread().getId();
            CasLog.casLog.log[ix + 2] = BuddyAllocator.makeIntPair(arenaIx, from);
            CasLog.casLog.log[ix + 3] = BuddyAllocator.makeIntPair(arenaIx, to);
        }

        public static void logSet(Src src, int arenaIx, int headerIndex, int identityHashCode) {
            if (casLog == null) {
                return;
            }
            int ix = CasLog.casLog.offset.addAndGet(4) - 4;
            CasLog.casLog.log[ix] = BuddyAllocator.makeIntPair(2, src.ordinal());
            CasLog.casLog.log[ix + 1] = Thread.currentThread().getId();
            CasLog.casLog.log[ix + 2] = BuddyAllocator.makeIntPair(arenaIx, headerIndex);
            CasLog.casLog.log[ix + 3] = identityHashCode;
        }

        public static void logRemoveFromList(int arenaIx, int headerIx, int listIx, int listHead) {
            if (casLog == null) {
                return;
            }
            int ix = CasLog.casLog.offset.addAndGet(4) - 4;
            CasLog.casLog.log[ix] = BuddyAllocator.makeIntPair(5, listIx);
            CasLog.casLog.log[ix + 1] = Thread.currentThread().getId();
            CasLog.casLog.log[ix + 2] = BuddyAllocator.makeIntPair(arenaIx, headerIx);
            CasLog.casLog.log[ix + 3] = listHead;
        }

        public static void logAddToList(int arenaIx, int headerIx, int listIx, int listHead) {
            if (casLog == null) {
                return;
            }
            int ix = CasLog.casLog.offset.addAndGet(4) - 4;
            CasLog.casLog.log[ix] = BuddyAllocator.makeIntPair(4, listIx);
            CasLog.casLog.log[ix + 1] = Thread.currentThread().getId();
            CasLog.casLog.log[ix + 2] = BuddyAllocator.makeIntPair(arenaIx, headerIx);
            CasLog.casLog.log[ix + 3] = listHead;
        }

        public static void logError() {
            if (casLog == null) {
                return;
            }
            int ix = CasLog.casLog.offset.addAndGet(2) - 2;
            CasLog.casLog.log[ix] = BuddyAllocator.makeIntPair(6, 0);
            CasLog.casLog.log[ix + 1] = Thread.currentThread().getId();
        }

        private int dumpOneLine(int ix) {
            int event = BuddyAllocator.getFirstInt(this.log[ix]);
            switch (event) {
                case 0: {
                    LlapIoImpl.LOG.info(this.prefix(ix) + " started to move " + this.header(this.log[ix + 2]) + " " + Integer.toHexString(BuddyAllocator.getSecondInt(this.log[ix])));
                    return ix + 3;
                }
                case 1: {
                    LlapIoImpl.LOG.info(this.prefix(ix) + " " + this.src(BuddyAllocator.getSecondInt(this.log[ix])) + " set " + this.header(this.log[ix + 2]) + " to taken of size " + this.log[ix + 3]);
                    return ix + 4;
                }
                case 3: {
                    LlapIoImpl.LOG.info(this.prefix(ix) + " " + this.src(BuddyAllocator.getSecondInt(this.log[ix])) + " set " + this.header(this.log[ix + 2]) + " to free of size " + this.log[ix + 3]);
                    return ix + 4;
                }
                case 7: {
                    LlapIoImpl.LOG.info(this.prefix(ix) + " " + this.src(BuddyAllocator.getSecondInt(this.log[ix])) + " unset [" + this.header(this.log[ix + 2]) + ", " + this.header(this.log[ix + 3]) + "]");
                    return ix + 4;
                }
                case 2: {
                    LlapIoImpl.LOG.info(this.prefix(ix) + " " + this.src(BuddyAllocator.getSecondInt(this.log[ix])) + " set " + this.header(this.log[ix + 2]) + " to " + Integer.toHexString((int)this.log[ix + 3]));
                    return ix + 4;
                }
                case 4: {
                    return ix + 4;
                }
                case 5: {
                    return ix + 4;
                }
                case 6: {
                    LlapIoImpl.LOG.error(this.prefix(ix) + " failed");
                    return ix + 2;
                }
            }
            throw new AssertionError((Object)("Unknown " + event));
        }

        private String prefix(int ix) {
            return ix + " thread-" + this.log[ix + 1];
        }

        private String src(int val) {
            return Src.values()[val].name();
        }

        private String header(long l) {
            return BuddyAllocator.getFirstInt(l) + ":" + BuddyAllocator.getSecondInt(l);
        }

        public synchronized void dumpLog(boolean doSleep) {
            if (doSleep) {
                try {
                    Thread.sleep(100L);
                }
                catch (InterruptedException interruptedException) {
                    // empty catch block
                }
            }
            int logSize = this.offset.get();
            int ix = 0;
            while (ix < logSize) {
                ix = this.dumpOneLine(ix);
            }
            this.offset.set(0);
        }

        public static enum Src {
            NEWLY_CLEARED,
            SPLIT_AFTER_BUF,
            SPLIT_AFTER_DEFRAG,
            ALLOC_SPLIT_DEFRAG,
            ALLOC_SPLIT_BUF,
            ALLOC_DEFRAG,
            EMPTY_V,
            NEW_BASE,
            CTOR,
            MOVE_TO_NESTED,
            MOVE_TO_ALLOC,
            ABANDON_MOVE,
            ABANDON_AT_END,
            ABANDON_BASE,
            CLEARED_BASE,
            CLEARED_VICTIM,
            UNUSABLE_NESTED,
            ABANDON_NESTED,
            DEALLOC,
            ALLOC_FREE_DEFRAG,
            ALLOC_FREE_BUF;

        }
    }

    private static class FreeList {
        ReentrantLock lock = new ReentrantLock(false);
        int listHead = -1;

        private FreeList() {
        }
    }

    private class Arena {
        private static final int FAILED_TO_RESERVE = Integer.MAX_VALUE;
        private int arenaIx;
        private ByteBuffer data;
        private LlapAllocatorBuffer[] buffers;
        private byte[] headers;
        private FreeList[] freeLists;

        private Arena() {
        }

        void init(int arenaIx) {
            this.arenaIx = arenaIx;
            try {
                this.data = BuddyAllocator.this.preallocateArenaBuffer(BuddyAllocator.this.arenaSize);
            }
            catch (OutOfMemoryError oom) {
                throw new OutOfMemoryError("Cannot allocate " + BuddyAllocator.this.arenaSize + " bytes: " + oom.getMessage() + "; make sure your xmx and process size are set correctly.");
            }
            int maxMinAllocs = 1 << BuddyAllocator.this.arenaSizeLog2 - BuddyAllocator.this.minAllocLog2;
            this.buffers = new LlapAllocatorBuffer[maxMinAllocs];
            this.headers = new byte[maxMinAllocs];
            int allocLog2Diff = BuddyAllocator.this.maxAllocLog2 - BuddyAllocator.this.minAllocLog2;
            int freeListCount = allocLog2Diff + 1;
            this.freeLists = new FreeList[freeListCount];
            for (int i = 0; i < freeListCount; ++i) {
                this.freeLists[i] = new FreeList();
            }
            int maxMaxAllocs = 1 << BuddyAllocator.this.arenaSizeLog2 - BuddyAllocator.this.maxAllocLog2;
            int headerIndex = 0;
            int headerStep = 1 << allocLog2Diff;
            this.freeLists[allocLog2Diff].listHead = 0;
            int i = 0;
            int offset = 0;
            while (i < maxMaxAllocs) {
                this.setHeaderFree(headerIndex, allocLog2Diff, CasLog.Src.CTOR);
                this.data.putInt(offset, i == 0 ? -1 : headerIndex - headerStep);
                this.data.putInt(offset + 4, i == maxMaxAllocs - 1 ? -1 : headerIndex + headerStep);
                headerIndex += headerStep;
                ++i;
                offset += BuddyAllocator.this.maxAllocation;
            }
        }

        public void checkHeader(int headerIx, int freeListIx, boolean isLocked) {
            BuddyAllocator.this.checkHeaderByte(this.arenaIx, headerIx, freeListIx, isLocked, this.headers[headerIx]);
        }

        public void reserveDiscardBruteForce(int freeListIx, DiscardContext ctx, int startHeaderIx) {
            if (this.data == null) {
                return;
            }
            int headerStep = 1 << freeListIx;
            int headerIx = startHeaderIx;
            do {
                long reserveResult = this.reserveBlockContents(freeListIx, headerIx, ctx.victimHeaders, ctx.victimCount, true);
                int reservedCount = BuddyAllocator.getFirstInt(reserveResult);
                int moveSize = BuddyAllocator.getSecondInt(reserveResult);
                if (moveSize == Integer.MAX_VALUE) {
                    for (int i = ctx.victimCount; i < ctx.victimCount + reservedCount; ++i) {
                        this.abandonOneHeaderBeingMoved(ctx.victimHeaders[i], CasLog.Src.ABANDON_MOVE);
                    }
                } else {
                    ctx.victimCount += reservedCount;
                    ctx.addBaseHeader(headerIx);
                }
                headerIx = BuddyAllocator.getNextIx(headerIx, this.headers.length, headerStep);
            } while (ctx.remainingToFind > 0 && headerIx != startHeaderIx);
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        public void reserveDiscardBlocksBasedOnFreeList(int mergeListIx, DiscardContext ctx) {
            if (this.data == null) {
                return;
            }
            FreeList freeList = this.freeLists[mergeListIx];
            freeList.lock.lock();
            try {
                int freeHeaderIx = freeList.listHead;
                while (freeHeaderIx >= 0) {
                    boolean reserved = false;
                    if (ctx.remainingToFind > 0) {
                        int headerToFreeIx = BuddyAllocator.getBuddyHeaderIx(mergeListIx, freeHeaderIx);
                        long reserveResult = this.reserveBlockContents(mergeListIx, headerToFreeIx, ctx.victimHeaders, ctx.victimCount, true);
                        int reservedCount = BuddyAllocator.getFirstInt(reserveResult);
                        int moveSize = BuddyAllocator.getSecondInt(reserveResult);
                        boolean bl = reserved = moveSize != Integer.MAX_VALUE;
                        if (!reserved) {
                            this.prepareAbandonUnfinishedMoveAttempt(ctx, reservedCount);
                        } else {
                            ctx.victimCount += reservedCount;
                            ctx.addBaseHeader(freeHeaderIx);
                        }
                    }
                    int nextFreeHeaderIx = this.getNextFreeListItem(BuddyAllocator.this.offsetFromHeaderIndex(freeHeaderIx));
                    if (reserved) {
                        this.removeBlockFromFreeList(freeList, freeHeaderIx, mergeListIx);
                        if (assertsEnabled) {
                            this.checkHeader(freeHeaderIx, mergeListIx, false);
                        }
                        this.setHeaderNoBufAlloc(freeHeaderIx, mergeListIx, CasLog.Src.NEW_BASE);
                    }
                    if (ctx.remainingToFind == 0) {
                        break;
                    }
                    freeHeaderIx = nextFreeHeaderIx;
                }
            }
            finally {
                freeList.lock.unlock();
            }
            for (int i = 0; i < ctx.abandonedCount; ++i) {
                this.abandonOneHeaderBeingMoved(ctx.abandonedHeaders[i], CasLog.Src.ABANDON_AT_END);
            }
            ctx.abandonedCount = 0;
        }

        private void prepareAbandonUnfinishedMoveAttempt(DiscardContext ctx, int count) {
            int start;
            if (count == 0) {
                return;
            }
            int startIx = ctx.victimCount;
            if (ctx.abandonedHeaders == null) {
                start = 0;
                ctx.abandonedHeaders = new int[count];
            } else {
                start = ctx.abandonedCount;
                int newLen = start + count;
                if (newLen > ctx.abandonedHeaders.length) {
                    ctx.abandonedHeaders = Arrays.copyOf(ctx.abandonedHeaders, newLen);
                }
            }
            System.arraycopy(ctx.victimHeaders, startIx, ctx.abandonedHeaders, start, count);
            ctx.abandonedCount += count;
            ctx.victimCount = startIx;
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        private long reserveBlockContents(int freeListIx, int headerToFreeIx, int[] victimHeaders, int victimsOffset, boolean isDiscard) {
            if (BuddyAllocator.this.enableDefragShortcut) {
                LlapAllocatorBuffer buffer = this.buffers[headerToFreeIx];
                byte header = this.headers[headerToFreeIx];
                if (buffer != null && BuddyAllocator.freeListFromHeader(header) == freeListIx) {
                    FreeList freeList = this.freeLists[freeListIx];
                    freeList.lock.lock();
                    try {
                        if (this.headers[headerToFreeIx] == header && buffer.startMoveOrDiscard(this.arenaIx, headerToFreeIx, isDiscard)) {
                            if (assertsEnabled) {
                                BuddyAllocator.this.assertBufferLooksValid(freeListIx, buffer, this.arenaIx, headerToFreeIx);
                                CasLog.logMove(this.arenaIx, headerToFreeIx, System.identityHashCode(buffer));
                            }
                            victimHeaders[victimsOffset] = headerToFreeIx;
                            long l = BuddyAllocator.makeIntPair(1, buffer.allocSize);
                            return l;
                        }
                    }
                    finally {
                        freeList.lock.unlock();
                    }
                }
            }
            long[] stack = new long[freeListIx + 1];
            int stackSize = 1;
            stack[0] = BuddyAllocator.makeIntPair(freeListIx, BuddyAllocator.getBuddyHeaderIx(freeListIx, headerToFreeIx));
            int victimCount = 0;
            int totalMoveSize = 0;
            while (stackSize > 0) {
                int sourceHeaderIx;
                long next;
                int listLevel;
                int levelBuddyHeaderIx;
                long result;
                if ((result = this.prepareOneHeaderForMove(levelBuddyHeaderIx = BuddyAllocator.getBuddyHeaderIx(listLevel = BuddyAllocator.getFirstInt(next = stack[--stackSize]), sourceHeaderIx = BuddyAllocator.getSecondInt(next)), isDiscard, freeListIx)) == -1L) {
                    return BuddyAllocator.makeIntPair(victimCount, Integer.MAX_VALUE);
                }
                int allocSize = BuddyAllocator.getFirstInt(result);
                totalMoveSize += allocSize;
                victimHeaders[victimsOffset + victimCount] = levelBuddyHeaderIx;
                ++victimCount;
                int actualBuddyListIx = BuddyAllocator.getSecondInt(result);
                for (int intermListIx = listLevel - 1; intermListIx >= actualBuddyListIx; --intermListIx) {
                    stack[stackSize++] = BuddyAllocator.makeIntPair(intermListIx, levelBuddyHeaderIx);
                }
            }
            return BuddyAllocator.makeIntPair(victimCount, totalMoveSize);
        }

        private void abandonOneHeaderBeingMoved(int headerIx, CasLog.Src src) {
            LlapAllocatorBuffer buf;
            byte header = this.headers[headerIx];
            int freeListIx = BuddyAllocator.freeListFromHeader(header);
            if ((header & 1) != 1) {
                BuddyAllocator.this.failWithLog("Victim header not in use");
            }
            if ((buf = this.buffers[headerIx]) != null) {
                if (assertsEnabled) {
                    BuddyAllocator.this.assertBufferLooksValid(freeListIx, buf, this.arenaIx, headerIx);
                }
                BuddyAllocator.this.cancelDiscard(buf, this.arenaIx, headerIx);
            } else {
                if (assertsEnabled) {
                    this.checkHeader(headerIx, -1, true);
                }
                this.addToFreeListWithMerge(headerIx, freeListIx, null, src);
            }
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        private long prepareOneHeaderForMove(int victimHeaderIx, boolean isDiscard, int maxListIx) {
            byte header = this.headers[victimHeaderIx];
            if (header == 0) {
                return -1L;
            }
            int freeListIx = BuddyAllocator.freeListFromHeader(header);
            if (freeListIx > maxListIx) {
                return -1L;
            }
            if (this.buffers[victimHeaderIx] == null && (header & 1) == 1) {
                return -1L;
            }
            FreeList freeList = this.freeLists[freeListIx];
            freeList.lock.lock();
            try {
                if (this.headers[victimHeaderIx] != header) {
                    long l = -1L;
                    return l;
                }
                LlapAllocatorBuffer buffer = this.buffers[victimHeaderIx];
                if (buffer == null && (header & 1) == 1) {
                    long l = -1L;
                    return l;
                }
                int allocSize = 0;
                if (buffer != null) {
                    if (!buffer.startMoveOrDiscard(this.arenaIx, victimHeaderIx, isDiscard)) {
                        long l = -1L;
                        return l;
                    }
                    CasLog.logMove(this.arenaIx, victimHeaderIx, System.identityHashCode(buffer));
                    allocSize = BuddyAllocator.this.allocSizeFromFreeList(freeListIx);
                } else {
                    this.setHeaderNoBufAlloc(victimHeaderIx, freeListIx, CasLog.Src.EMPTY_V);
                    this.removeBlockFromFreeList(freeList, victimHeaderIx, freeListIx);
                }
                long l = BuddyAllocator.makeIntPair(allocSize, freeListIx);
                return l;
            }
            finally {
                freeList.lock.unlock();
            }
        }

        public int allocateFromDiscard(MemoryBuffer[] dest, int destIx, int headerIx, int freeListIx, int allocationSize) {
            LlapAllocatorBuffer buffer = (LlapAllocatorBuffer)dest[destIx];
            this.initializeNewlyAllocated(buffer, allocationSize, headerIx, BuddyAllocator.this.offsetFromHeaderIndex(headerIx));
            if (assertsEnabled) {
                this.checkHeader(headerIx, freeListIx, true);
            }
            this.setHeaderAlloc(headerIx, freeListIx, buffer, CasLog.Src.ALLOC_DEFRAG);
            return destIx + 1;
        }

        private void setHeaderAlloc(int headerIx, int freeListIx, LlapAllocatorBuffer alloc, CasLog.Src src) {
            assert (alloc != null);
            this.headers[headerIx] = BuddyAllocator.makeHeader(freeListIx, true);
            this.buffers[headerIx] = alloc;
            CasLog.logSet(src, this.arenaIx, headerIx, System.identityHashCode(alloc));
        }

        private void setHeaderFree(int headerIndex, int freeListIx, CasLog.Src src) {
            this.headers[headerIndex] = BuddyAllocator.makeHeader(freeListIx, false);
            this.buffers[headerIndex] = null;
            CasLog.logSetFree(src, this.arenaIx, headerIndex, BuddyAllocator.this.allocSizeFromFreeList(freeListIx));
        }

        private void setHeaderNoBufAlloc(int headerIndex, int freeListIx, CasLog.Src src) {
            this.headers[headerIndex] = BuddyAllocator.makeHeader(freeListIx, true);
            CasLog.logSetNb(src, this.arenaIx, headerIndex, BuddyAllocator.this.allocSizeFromFreeList(freeListIx));
        }

        private void unsetHeader(int headerIndex, CasLog.Src src) {
            this.headers[headerIndex] = 0;
            CasLog.logUnset(src, this.arenaIx, headerIndex, headerIndex);
        }

        private void unsetHeaders(int fromHeaderIx, int toHeaderIx, CasLog.Src src) {
            Arrays.fill(this.headers, fromHeaderIx, toHeaderIx, (byte)0);
            CasLog.logUnset(src, this.arenaIx, fromHeaderIx, toHeaderIx - 1);
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        private void debugDump(StringBuilder result) {
            result.append("\nArena: ");
            if (this.data == null) {
                result.append(" not allocated");
                return;
            }
            byte[] headers = new byte[this.headers.length];
            System.arraycopy(this.headers, 0, headers, 0, headers.length);
            int allocSize = BuddyAllocator.this.minAllocation;
            int i = 0;
            while (i < this.freeLists.length) {
                result.append("\n  free list for size " + allocSize + ": ");
                FreeList freeList = this.freeLists[i];
                freeList.lock.lock();
                try {
                    int nextHeaderIx = freeList.listHead;
                    while (nextHeaderIx >= 0) {
                        result.append(nextHeaderIx + ", ");
                        nextHeaderIx = this.getNextFreeListItem(BuddyAllocator.this.offsetFromHeaderIndex(nextHeaderIx));
                    }
                }
                finally {
                    freeList.lock.unlock();
                }
                ++i;
                allocSize <<= 1;
            }
            for (i = 0; i < headers.length; ++i) {
                byte header = headers[i];
                if (header == 0) continue;
                int freeListIx = BuddyAllocator.freeListFromHeader(header);
                int offset = BuddyAllocator.this.offsetFromHeaderIndex(i);
                boolean isFree = this.buffers[i] == null;
                result.append("\n  block " + i + " at " + offset + ": size " + (1 << freeListIx + BuddyAllocator.this.minAllocLog2) + ", " + (isFree ? "free" : "allocated"));
            }
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        public Integer debugDumpShort(StringBuilder result) {
            if (this.data == null) {
                return null;
            }
            int allocSize = BuddyAllocator.this.minAllocation;
            int total = 0;
            int i = 0;
            while (i < this.freeLists.length) {
                FreeList freeList = this.freeLists[i];
                freeList.lock.lock();
                try {
                    int nextHeaderIx = freeList.listHead;
                    int count = 0;
                    while (nextHeaderIx >= 0) {
                        ++count;
                        nextHeaderIx = this.getNextFreeListItem(BuddyAllocator.this.offsetFromHeaderIndex(nextHeaderIx));
                    }
                    if (count > 0) {
                        if (total == 0) {
                            result.append("\nArena with free list lengths by size: ");
                        }
                        total += allocSize * count;
                        result.append(allocSize).append(" => ").append(count).append(", ");
                    }
                }
                finally {
                    freeList.lock.unlock();
                }
                ++i;
                allocSize <<= 1;
            }
            return total;
        }

        private void testDump(StringBuilder result) {
            result.append("{");
            if (this.data == null) {
                result.append("}, ");
                return;
            }
            byte[] headers = new byte[this.headers.length];
            System.arraycopy(this.headers, 0, headers, 0, headers.length);
            for (int i = 0; i < headers.length; ++i) {
                byte header = headers[i];
                if (header == 0) continue;
                String allocState = ".";
                if (this.buffers[i] != null) {
                    allocState = "*";
                } else if ((header & 1) == 1) {
                    allocState = "!";
                }
                int size = 1 << BuddyAllocator.freeListFromHeader(header) + BuddyAllocator.this.minAllocLog2;
                result.append("[").append(size).append(allocState).append("@").append(i).append("]");
            }
            result.append("}, ");
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        private int allocateFast(int freeListIx, MemoryBuffer[] dest, long[] destHeaders, int destIx, int destCount, int allocSize) {
            if (this.data == null) {
                return -1;
            }
            FreeList freeList = this.freeLists[freeListIx];
            if (!freeList.lock.tryLock()) {
                return destIx;
            }
            try {
                int n = this.allocateFromFreeListUnderLock(freeList, freeListIx, dest, destHeaders, destIx, destCount, allocSize);
                return n;
            }
            finally {
                freeList.lock.unlock();
            }
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        private int allocateWithSplit(int freeListIx, MemoryBuffer[] dest, long[] destHeaders, int destIx, int destCount, int allocSize, int maxSplitFreeListIx) {
            if (this.data == null) {
                return -1;
            }
            FreeList freeList = this.freeLists[freeListIx];
            int remaining = -1;
            freeList.lock.lock();
            try {
                destIx = this.allocateFromFreeListUnderLock(freeList, freeListIx, dest, destHeaders, destIx, destCount, allocSize);
                remaining = destCount - destIx;
                if (remaining == 0) {
                    int n = destIx;
                    return n;
                }
            }
            finally {
                freeList.lock.unlock();
            }
            int headerStep = 1 << freeListIx;
            int splitListIx = freeListIx + 1;
            if (maxSplitFreeListIx == -1) {
                maxSplitFreeListIx = this.freeLists.length - 1;
            }
            while (remaining > 0 && splitListIx <= maxSplitFreeListIx) {
                CasLog.Src src;
                int splitWaysLog2 = splitListIx - freeListIx;
                assert (splitWaysLog2 > 0);
                int splitWays = 1 << splitWaysLog2;
                int lastSplitBlocksRemaining = -1;
                int lastSplitNextHeader = -1;
                FreeList splitList = this.freeLists[splitListIx];
                splitList.lock.lock();
                try {
                    int headerIx = splitList.listHead;
                    while (headerIx >= 0 && remaining > 0) {
                        int origOffset;
                        int offset = origOffset = BuddyAllocator.this.offsetFromHeaderIndex(headerIx);
                        int toTake = Math.min(splitWays, remaining);
                        remaining -= toTake;
                        lastSplitBlocksRemaining = splitWays - toTake;
                        while (toTake > 0) {
                            if (assertsEnabled) {
                                this.checkHeader(headerIx, -1, false);
                            }
                            if (dest != null) {
                                LlapAllocatorBuffer buffer = (LlapAllocatorBuffer)dest[destIx];
                                this.initializeNewlyAllocated(buffer, allocSize, headerIx, offset);
                                this.setHeaderAlloc(headerIx, freeListIx, buffer, CasLog.Src.ALLOC_SPLIT_BUF);
                            } else {
                                destHeaders[destIx] = BuddyAllocator.makeIntPair(this.arenaIx, headerIx);
                                this.setHeaderNoBufAlloc(headerIx, freeListIx, CasLog.Src.ALLOC_SPLIT_DEFRAG);
                            }
                            ++destIx;
                            --toTake;
                            headerIx += headerStep;
                            offset += allocSize;
                        }
                        lastSplitNextHeader = headerIx;
                        headerIx = this.getNextFreeListItem(origOffset);
                    }
                    this.replaceListHeadUnderLock(splitList, headerIx, splitListIx);
                }
                finally {
                    splitList.lock.unlock();
                }
                CasLog.Src src2 = src = dest != null ? CasLog.Src.SPLIT_AFTER_BUF : CasLog.Src.SPLIT_AFTER_DEFRAG;
                if (remaining == 0) {
                    int newListIndex = freeListIx;
                    while (lastSplitBlocksRemaining > 0) {
                        if ((lastSplitBlocksRemaining & 1) == 1) {
                            this.addToFreeListWithMerge(lastSplitNextHeader, newListIndex, null, src);
                            lastSplitNextHeader += 1 << newListIndex;
                        }
                        lastSplitBlocksRemaining >>>= 1;
                        ++newListIndex;
                    }
                }
                ++splitListIx;
            }
            return destIx;
        }

        private void initializeNewlyAllocated(LlapAllocatorBuffer buffer, int allocSize, int headerIx, int offset) {
            buffer.initialize(this.data, offset, allocSize);
            buffer.setNewAllocLocation(this.arenaIx, headerIx);
        }

        private void replaceListHeadUnderLock(FreeList freeList, int headerIx, int ix) {
            if (headerIx == freeList.listHead) {
                return;
            }
            if (headerIx >= 0) {
                int newHeadOffset = BuddyAllocator.this.offsetFromHeaderIndex(headerIx);
                this.data.putInt(newHeadOffset, -1);
            }
            freeList.listHead = headerIx;
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        private int allocateWithExpand(int arenaIx, int freeListIx, MemoryBuffer[] dest, int ix, int size) {
            int arenaCount;
            while (true) {
                int allocArenaCount = arenaCount = BuddyAllocator.this.allocatedArenas.get();
                if (arenaCount < 0) {
                    allocArenaCount = -arenaCount - 1;
                }
                if (allocArenaCount > arenaIx) {
                    return this.allocateWithSplit(freeListIx, dest, null, ix, dest.length, size, -1);
                }
                if (arenaIx + 1 == -arenaCount) {
                    try {
                        Arena arena = this;
                        synchronized (arena) {
                            this.wait(100L);
                        }
                    }
                    catch (InterruptedException e) {
                        Thread.currentThread().interrupt();
                    }
                    continue;
                }
                assert (arenaCount == arenaIx) : "Arena count " + arenaCount + " but " + arenaIx + " is not being allocated";
                if (BuddyAllocator.this.allocatedArenas.compareAndSet(arenaCount, -arenaCount - 1)) break;
            }
            assert (this.data == null);
            this.init(arenaIx);
            boolean isCommited = BuddyAllocator.this.allocatedArenas.compareAndSet(-arenaCount - 1, arenaCount + 1);
            assert (isCommited);
            Arena arena = this;
            synchronized (arena) {
                this.notifyAll();
            }
            BuddyAllocator.this.metrics.incrAllocatedArena();
            return this.allocateWithSplit(freeListIx, dest, null, ix, dest.length, size, -1);
        }

        public int allocateFromFreeListUnderLock(FreeList freeList, int freeListIx, MemoryBuffer[] dest, long[] destHeaders, int destIx, int destCount, int allocSize) {
            int current = freeList.listHead;
            assert (dest == null != (destHeaders == null));
            while (current >= 0 && destIx < destCount) {
                int offset = BuddyAllocator.this.offsetFromHeaderIndex(current);
                int allocHeaderIx = current;
                current = this.getNextFreeListItem(offset);
                if (assertsEnabled) {
                    this.checkHeader(allocHeaderIx, freeListIx, false);
                }
                if (dest != null) {
                    LlapAllocatorBuffer buffer = (LlapAllocatorBuffer)dest[destIx];
                    this.initializeNewlyAllocated(buffer, allocSize, allocHeaderIx, offset);
                    this.setHeaderAlloc(allocHeaderIx, freeListIx, buffer, CasLog.Src.ALLOC_FREE_BUF);
                } else {
                    destHeaders[destIx] = BuddyAllocator.makeIntPair(this.arenaIx, allocHeaderIx);
                    this.setHeaderNoBufAlloc(allocHeaderIx, freeListIx, CasLog.Src.ALLOC_FREE_DEFRAG);
                }
                ++destIx;
            }
            this.replaceListHeadUnderLock(freeList, current, freeListIx);
            return destIx;
        }

        private int getPrevFreeListItem(int offset) {
            return this.data.getInt(offset);
        }

        private int getNextFreeListItem(int offset) {
            return this.data.getInt(offset + 4);
        }

        public void deallocate(LlapAllocatorBuffer buffer, boolean isAfterMove) {
            assert (this.data != null);
            int pos = buffer.byteBuffer.position();
            int headerIx = pos >>> BuddyAllocator.this.minAllocLog2;
            int freeListIx = BuddyAllocator.this.freeListFromAllocSize(buffer.allocSize);
            if (assertsEnabled && !isAfterMove) {
                LlapAllocatorBuffer buf = this.buffers[headerIx];
                if (buf != buffer) {
                    BuddyAllocator.this.failWithLog(this.arenaIx + ":" + headerIx + " => " + BuddyAllocator.toDebugString(buffer) + ", " + BuddyAllocator.toDebugString(buf));
                }
                BuddyAllocator.this.assertBufferLooksValid(BuddyAllocator.freeListFromHeader(this.headers[headerIx]), buf, this.arenaIx, headerIx);
                this.checkHeader(headerIx, freeListIx, true);
            }
            this.buffers[headerIx] = null;
            this.addToFreeListWithMerge(headerIx, freeListIx, buffer, CasLog.Src.DEALLOC);
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        private void addToFreeListWithMerge(int headerIx, int freeListIx, LlapAllocatorBuffer buffer, CasLog.Src src) {
            while (true) {
                FreeList freeList = this.freeLists[freeListIx];
                int bHeaderIx = BuddyAllocator.getBuddyHeaderIx(freeListIx, headerIx);
                freeList.lock.lock();
                try {
                    if (freeListIx == this.freeLists.length - 1 || this.headers[bHeaderIx] != BuddyAllocator.makeHeader(freeListIx, false)) {
                        this.addBlockToFreeListUnderLock(freeList, headerIx, freeListIx);
                        this.setHeaderFree(headerIx, freeListIx, src);
                        break;
                    }
                    this.removeBlockFromFreeList(freeList, bHeaderIx, freeListIx);
                    this.unsetHeader(bHeaderIx, src);
                    this.unsetHeader(headerIx, src);
                }
                finally {
                    freeList.lock.unlock();
                }
                ++freeListIx;
                headerIx = Math.min(headerIx, bHeaderIx);
            }
        }

        private void addBlockToFreeListUnderLock(FreeList freeList, int headerIx, int ix) {
            CasLog.logAddToList(this.arenaIx, headerIx, ix, freeList.listHead);
            if (freeList.listHead >= 0) {
                int oldHeadOffset = BuddyAllocator.this.offsetFromHeaderIndex(freeList.listHead);
                assert (this.getPrevFreeListItem(oldHeadOffset) == -1);
                this.data.putInt(oldHeadOffset, headerIx);
            }
            int offset = BuddyAllocator.this.offsetFromHeaderIndex(headerIx);
            this.data.putInt(offset, -1);
            this.data.putInt(offset + 4, freeList.listHead);
            freeList.listHead = headerIx;
        }

        private void removeBlockFromFreeList(FreeList freeList, int headerIx, int ix) {
            int bOffset = BuddyAllocator.this.offsetFromHeaderIndex(headerIx);
            int bpHeaderIx = this.getPrevFreeListItem(bOffset);
            int bnHeaderIx = this.getNextFreeListItem(bOffset);
            CasLog.logRemoveFromList(this.arenaIx, headerIx, ix, freeList.listHead);
            if (freeList.listHead == headerIx) {
                assert (bpHeaderIx == -1);
                freeList.listHead = bnHeaderIx;
            }
            if (bpHeaderIx != -1) {
                this.data.putInt(BuddyAllocator.this.offsetFromHeaderIndex(bpHeaderIx) + 4, bnHeaderIx);
            }
            if (bnHeaderIx != -1) {
                this.data.putInt(BuddyAllocator.this.offsetFromHeaderIndex(bnHeaderIx), bpHeaderIx);
            }
        }
    }

    private static final class DiscardContext {
        long[] results;
        int resultCount;
        int memoryReleased;
        int[] victimHeaders;
        int victimCount;
        int[] baseHeaders;
        int baseCount;
        int remainingToFind;
        int[] abandonedHeaders;
        int abandonedCount;

        private DiscardContext() {
        }

        void init(int headersPerOneReq, int reqCount) {
            this.resetResults();
            this.remainingToFind = reqCount;
            if (this.results == null || this.results.length < reqCount) {
                this.results = new long[reqCount];
                this.baseHeaders = new int[reqCount];
            }
            int maxVictimCount = headersPerOneReq * reqCount;
            if (this.victimHeaders == null || this.victimHeaders.length < maxVictimCount) {
                this.victimHeaders = new int[maxVictimCount];
            }
        }

        void resetResults() {
            this.resetBetweenArenas();
            this.memoryReleased = 0;
            this.resultCount = 0;
        }

        void resetBetweenArenas() {
            this.abandonedCount = 0;
            this.baseCount = 0;
            this.victimCount = 0;
        }

        public void addResult(int arenaIx, int freeHeaderIx) {
            this.results[this.resultCount] = BuddyAllocator.makeIntPair(arenaIx, freeHeaderIx);
            ++this.resultCount;
        }

        public void addBaseHeader(int headerIx) {
            this.baseHeaders[this.baseCount] = headerIx;
            ++this.baseCount;
            --this.remainingToFind;
        }

        public String toString() {
            return "[victimHeaders=" + Arrays.toString(this.victimHeaders) + ", victimCount=" + this.victimCount + ", baseHeaders=" + Arrays.toString(this.baseHeaders) + ", baseCount=" + this.baseCount + ", remainingToFind=" + this.remainingToFind + "]";
        }
    }
}

