package com.alibaba.schedulerx.worker.security;

import java.util.List;
import java.util.Map;

import com.alibaba.schedulerx.common.domain.JSONResult;
import com.alibaba.schedulerx.common.util.ConfigUtil;
import com.alibaba.schedulerx.common.util.JsonUtil;
import com.alibaba.schedulerx.worker.domain.WorkerConstants;
import com.alibaba.schedulerx.worker.exception.AuthenticateException;
import com.alibaba.schedulerx.worker.log.LogFactory;
import com.alibaba.schedulerx.worker.log.Logger;

import com.google.common.collect.Maps;
import com.mashape.unirest.http.HttpResponse;
import com.mashape.unirest.http.JsonNode;
import com.mashape.unirest.http.Unirest;
import com.mashape.unirest.http.exceptions.UnirestException;
import com.taobao.spas.sdk.client.SpasSdkClientFacade;
import com.taobao.spas.sdk.client.identity.Credentials;
import org.apache.commons.configuration.Configuration;
import org.apache.commons.lang.StringUtils;

/**
 *
 * @author 陆昆
 */
public class AliyunAuthenticator implements Authenticator {
    private static final String AUTHENTICATE_DEFAULT = "authenticate_default";
    private static final String AUTHENTICATE_DAUTH = "authenticate_dauth";
    private static final String AUTHENTICATE_RAM = "authenticate_ram";
    private static final String AUTHENTICATE_STS = "authenticate_sts";
    private static final String AUTHENTICATE_URL = "/worker/v1/appgroup/authenticate";
    private static final Logger LOGGER = LogFactory.getLogger(AliyunAuthenticator.class); 

    @Override
    public void authenticate(Configuration conf, String namespace, String namespaceSource, List<String> groupIds)
            throws AuthenticateException {
        JSONResult result = JSONResult.geneFailResult();
        String accessKey = StringUtils.EMPTY;
        String secretKey = StringUtils.EMPTY;
        String secretToken = StringUtils.EMPTY;
        
        // -Dschedulerx.appKey优先
        String appKeyProperty = System.getProperty(WorkerConstants.WORKER_APPKEY);
        if (StringUtils.isNotBlank(appKeyProperty)) {
            conf.setProperty(WorkerConstants.APP_KEY, appKeyProperty);
        }
        String[] appKeys = conf.getStringArray(WorkerConstants.APP_KEY);
        Map<String, Object> parameterMap = Maps.newHashMap();
        if (StringUtils.isNotBlank(namespace)) {
            parameterMap.put("namespace", namespace);
        }
        if (StringUtils.isNotBlank(namespaceSource)) {
            parameterMap.put("namespaceSource", namespaceSource);
        }
        parameterMap.put("groups", StringUtils.join(groupIds,","));
        
        //appKey鉴权
        if (appKeys != null && appKeys.length > 0) {
            parameterMap.put("accessKey", StringUtils.join(appKeys,","));
            parameterMap.put("type", AUTHENTICATE_DEFAULT);
        } else {
            //阿里云ak/sk鉴权
            if (StringUtils.isNotBlank(conf.getString(WorkerConstants.ALIYUN_ACESSKEY))) {
                accessKey = conf.getString(WorkerConstants.ALIYUN_ACESSKEY);
                secretKey = conf.getString(WorkerConstants.ALIYUN_SECRETKEY);
                parameterMap.put("type", AUTHENTICATE_RAM);
            } else if (StringUtils.isNotBlank(conf.getString(WorkerConstants.STS_ACESSKEY))) {
                accessKey = conf.getString(WorkerConstants.STS_ACESSKEY);
                secretKey = conf.getString(WorkerConstants.STS_SECRETKEY);
                secretToken = conf.getString(WorkerConstants.STS_TOKEN);
                parameterMap.put("secretToken", secretToken);
                parameterMap.put("type", AUTHENTICATE_STS);
            } else {
                // 从dauth中获取ak/sk（EDAS）
                Credentials credentials = SpasSdkClientFacade.getCredential();
                if (StringUtils.isNotBlank(credentials.getAccessKey())) {
                    accessKey = credentials.getAccessKey();
                }
                if (StringUtils.isNotBlank(credentials.getSecretKey())) {
                    secretKey = credentials.getSecretKey();
                }
                // 默认dauth为edas来源
                conf.setProperty(WorkerConstants.WORKER_SOURCE, WorkerConstants.WORKER_SOURCE_EDAS);
                parameterMap.put("type", AUTHENTICATE_DAUTH);
            }
            
            if (StringUtils.isBlank(accessKey)) {
                throw new AuthenticateException("ak is blank");
            }
            if (StringUtils.isBlank(secretKey)) {
                throw new AuthenticateException("sk is blank");
            }
            
            parameterMap.put("accessKey", accessKey);
            parameterMap.put("secretKey", secretKey);
        }
        
        String domain = ConfigUtil.getWorkerConfig().getString(WorkerConstants.WORKER_DOMAIN_NAME);
        if (StringUtils.isNotBlank(domain)) {
            String url = "http://" + domain + AUTHENTICATE_URL;
            try {
                HttpResponse<JsonNode> jsonResponse = Unirest.post(url).fields(parameterMap).asJson();
                if (jsonResponse != null) {
                    result = JsonUtil.fromJson(jsonResponse.getBody().toString(), JSONResult.class);
                }
            } catch (UnirestException e) {
                LOGGER.error("groupIds: {} authenticate error, url={}", groupIds, url, e);
            } catch (Exception e) {
                LOGGER.error("groupIds: {} authenticate error, url={}", groupIds, url, e);
            }
        }
        
        if (result != null) {
            if (!result.isSuccess() || !(Boolean)result.getData()) {
                throw new AuthenticateException(result.getMessage());
            }
        } else {
            throw new AuthenticateException("authenticate result is null");
        }
    }
    
}
