package com.alibaba.schedulerx.worker.processor;

import java.io.BufferedReader;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.nio.charset.StandardCharsets;
import java.util.concurrent.CountDownLatch;

import com.alibaba.schedulerx.common.domain.StreamType;
import com.alibaba.schedulerx.common.util.ExceptionUtil;
import com.alibaba.schedulerx.worker.domain.JobContext;
import com.alibaba.schedulerx.worker.log.LogFactory;
import com.alibaba.schedulerx.worker.log.Logger;
import com.alibaba.schedulerx.worker.logcollector.LogCollector;
import com.alibaba.schedulerx.worker.logcollector.LogCollectorFactory;
import com.alibaba.schedulerx.worker.util.ShellUtil;

import org.apache.commons.lang.StringUtils;


/**
 *
 * @author xiaomeng.hxm
 */
public class ShellProcessor implements JobProcessor {

    private Process shellProcess = null;
    private LogCollector logCollector = LogCollectorFactory.get();
    protected static final Logger LOGGER = LogFactory.getLogger(ShellProcessor.class);
    private String uniqueId = null;

    @SuppressWarnings("finally")
    @Override
    public ProcessResult process(JobContext context) {
        ProcessResult result = new ProcessResult(false);
        logCollector.collect(uniqueId, "Script process start to run, taskName=" + context.getTaskName());
        try {
            ProcessBuilder processBuilder = ShellUtil.createProcessBuilder(getContent(context));
            if (redirectStream()) {
                processBuilder.redirectErrorStream(true);
            }
            shellProcess = processBuilder.start();

            CountDownLatch countDownLatch = null;
            if (redirectStream()) {
                countDownLatch = new CountDownLatch(1);
                Thread stderrStreamProcessor = new ShellStreamProcessor(this, shellProcess.getInputStream(),
                        StreamType.STD_ERR, countDownLatch);
                stderrStreamProcessor.start();
            } else {
                countDownLatch = new CountDownLatch(2);
                Thread stdoutStreamProcessor = new ShellStreamProcessor(this, shellProcess.getInputStream(),
                        StreamType.STD_OUT, countDownLatch);
                stdoutStreamProcessor.start();

                Thread stderrStreamProcessor = new ShellStreamProcessor(this, shellProcess.getErrorStream(),
                        StreamType.STD_ERR, countDownLatch);
                stderrStreamProcessor.start();
            }

            countDownLatch.await();
            if (shellProcess.waitFor() == 0) {
                result.setStatus(true);
            }
        } catch (Throwable e) {
            LOGGER.error("", e);
            logCollector.collect(uniqueId, "script process errors", e);
            result.setResult(ExceptionUtil.getMessage(e));
        } finally {
            logCollector.collect(uniqueId, "Script process end, run status=" + result.getStatus().getEnDesc());
            return result;
        }
    }

    protected String[] getContent(JobContext context) {
 //       return new String[] {"/bin/sh", "-c", context.getContent()};
        String[] parameters;
        if (StringUtils.isNotEmpty(context.getShardingParameter())) {
            parameters = new String[2];
            parameters[0] = String.valueOf(context.getShardingId());
            parameters[1] = context.getShardingParameter();
        } else if (StringUtils.isNotEmpty(context.getInstanceParameters())) {
            parameters = context.getInstanceParameters().trim().split(" ");
        } else {
            parameters = context.getJobParameters().trim().split(" ");
        }
        String[] contents = new String[3 + parameters.length];
        contents[0] = "/bin/sh";
        contents[1] = "-c";
        contents[2] = context.getContent();
        for (int i = 0; i < parameters.length; i++) {
            contents[3+i] = parameters[i];
        }
        return contents;
    }

    protected void processStdOutputStream(InputStream inputStream) {
        String line = null;
        try {
            BufferedReader br = new BufferedReader(new InputStreamReader(inputStream, StandardCharsets.UTF_8));
            while ((line = br.readLine()) != null) {
                logCollector.collect(uniqueId, line);
            }
        } catch (Throwable e) {
            LOGGER.error("error ShellJobProcessor stdout stream", e);
            logCollector.collect(uniqueId, "error process stdout stream", e);
        } finally {
            logCollector.collect(uniqueId, line, StreamType.STD_OUT, true);
        }
    }

    protected void processStdErrorStream(InputStream inputStream) {
        String line = null;
        try {
            BufferedReader br = new BufferedReader(new InputStreamReader(inputStream, StandardCharsets.UTF_8));
            while ((line = br.readLine()) != null) {
                logCollector.collect(uniqueId, line);
            }
        } catch (Throwable e) {
            LOGGER.error("error ShellJobProcessor stderr stream", e);
            logCollector.collect(uniqueId, "error process stderr stream", e);
        } finally {
            logCollector.collect(uniqueId, line, StreamType.STD_OUT, true);
        }
    }

    protected boolean redirectStream() {
        return true;
    }

    @Override
    public ProcessResult postProcess(JobContext context) {
        return null;
    }

    @Override
    public void kill(JobContext context) {
        try {
            long pid = ShellUtil.getPidOfProcess(shellProcess);
            if (pid > 0) {
                ShellUtil.killProcess(pid);
            }
        } catch (Throwable e) {
            LOGGER.error("kill shell job jobInstanceId={} failed, {}", context.getJobInstanceId(), e);
        }

        try {
            if (shellProcess != null) {
                shellProcess.destroy();
            }
        }catch (Throwable th) {
            LOGGER.error("kill shell job jobInstanceId={} failed, {}", context.getJobInstanceId(), th);
        }
    }

    @Override
    public void preProcess(JobContext context) throws Exception {
        uniqueId = context.getUniqueId();
    }

}
