package com.alibaba.schedulerx.worker.master.handler;

import java.util.Map;

import com.alibaba.schedulerx.common.constants.CommonConstants;
import com.alibaba.schedulerx.common.domain.InstanceStatus;
import com.alibaba.schedulerx.common.domain.JobInstanceInfo;
import com.alibaba.schedulerx.common.domain.LimitedQueue;
import com.alibaba.schedulerx.common.domain.ProgressHistory;
import com.alibaba.schedulerx.common.domain.SecondProgressDetail;
import com.alibaba.schedulerx.common.domain.TaskProgressCounter;
import com.alibaba.schedulerx.common.domain.WorkerProgressCounter;
import com.alibaba.schedulerx.common.util.IdUtil;
import com.alibaba.schedulerx.common.util.JsonUtil;
import com.alibaba.schedulerx.protocol.Worker.WorkerReportJobInstanceProgressRequest;
import com.alibaba.schedulerx.protocol.Worker.WorkerReportJobInstanceStatusRequest;
import com.alibaba.schedulerx.worker.SchedulerxWorker;
import com.alibaba.schedulerx.worker.discovery.ServerDiscoveryFactory;
import com.alibaba.schedulerx.worker.ha.HealthTimeHolder;
import com.alibaba.schedulerx.worker.log.LogFactory;
import com.alibaba.schedulerx.worker.log.Logger;
import com.alibaba.schedulerx.worker.logcollector.ClientLoggerMessage;
import com.alibaba.schedulerx.worker.logcollector.LogCollector;
import com.alibaba.schedulerx.worker.logcollector.LogCollectorFactory;
import com.alibaba.schedulerx.worker.master.BroadcastTaskMaster;
import com.alibaba.schedulerx.worker.master.MapTaskMaster;
import com.alibaba.schedulerx.worker.master.StandaloneTaskMaster;
import com.alibaba.schedulerx.worker.master.TaskMaster;
import com.alibaba.schedulerx.worker.processor.ProcessResult;
import com.alibaba.schedulerx.worker.util.ActorPathUtil;

import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.collections.MapUtils;
import org.apache.commons.lang.StringUtils;
import org.joda.time.DateTime;
import org.joda.time.format.DateTimeFormat;
import org.joda.time.format.DateTimeFormatter;

/**
 *
 * @author zhaibian
 * @version $Id: SecondJobUpdateInstanceStatusHandler.java, v 0.1 2019年02月28日 19:42 zhaibian Exp $
 */
public class SecondJobUpdateInstanceStatusHandler extends UpdateInstanceStatusHandler{

    private LogCollector logCollector = LogCollectorFactory.get();
    private static final Logger LOGGER = LogFactory.getLogger(SecondJobUpdateInstanceStatusHandler.class);


    private static final int MISS_SERVER_KILL_TIME = 30;

    SecondProgressDetail secondProgressDetail;

    LimitedQueue<ProgressHistory> recentProgressHistory = new LimitedQueue<>(10);

    long cycleStartTime = System.currentTimeMillis();

    SecondJobUpdateInstanceStatusHandler(TaskMaster taskMaster, JobInstanceInfo jobInstanceInfo) {
        super(taskMaster, jobInstanceInfo);
        secondProgressDetail = new SecondProgressDetail();
        init();
    }

    @Override
    public void handle(long serialNum, InstanceStatus instanceStatus, String result) throws Exception {
        String cycleId = IdUtil.getUniqueId(jobInstanceInfo.getJobId(), jobInstanceInfo.getJobInstanceId(), taskMaster.getSerialNum());
        LOGGER.info("cycleId: {} instanceStatus={} cycle update status.", cycleId, instanceStatus);

        // if init failed, instance status finished and master has not been killed, so second job kill self
        if (!taskMaster.isInited() && (instanceStatus.isFinish()) && !taskMaster.isKilled()) {
            taskMaster.killInstance("killed, because of worker init failed.");
            LOGGER.warn("Init failed need to kill self, cycleId={}", cycleId);
            return;
        }

        // if instance is killed, need to report to server
        if (taskMaster.isKilled() && StringUtils.contains(result, "killed")) {
            taskMaster.setInstanceStatus(InstanceStatus.FAILED);
            taskMaster.stop();
            masterPool.remove(jobInstanceInfo.getJobInstanceId());
            WorkerReportJobInstanceStatusRequest.Builder builder = WorkerReportJobInstanceStatusRequest
                    .newBuilder()
                    .setJobId(jobInstanceInfo.getJobId())
                    .setJobInstanceId(jobInstanceInfo.getJobInstanceId())
                    .setStatus(instanceStatus.getValue())
                    .setGroupId(jobInstanceInfo.getGroupId());

            if (result != null) {
                builder.setResult(result);
            }

            String progress = getJobInstanceProgress();
            if (progress != null) {
                builder.setProgress(progress);
            }
            SchedulerxWorker.AtLeastDeliveryRoutingActor.tell(builder.build(), null);
            LOGGER.info("report cycleId={}, status={} to AtLeastDeliveryRoutingActor", cycleId,
                instanceStatus);
        }

        //if job instance is finished, remove from TaskMasterPool
        if (instanceStatus.isFinish()) {
            triggerNextCycle(cycleId, serialNum, instanceStatus);
        }
    }

    private synchronized void triggerNextCycle(String cycleId, long serialNum, InstanceStatus instanceStatus) {
        if (serialNum != taskMaster.getSerialNum()) {
            LOGGER.info("triggerNextCycle={} ignore, current serialNum={}, but trigger serialNum={}, status={}, killed={}.",cycleId,
                taskMaster.getSerialNum(), serialNum, instanceStatus, taskMaster.isKilled());
            return;
        }

        ProcessResult postResult = taskMaster.postFinish(jobInstanceInfo.getJobInstanceId());
        if (postResult != null) {
            LOGGER.info("cycleId: {} cycle post status, result={}.", cycleId, postResult.getStatus(),
                postResult.getResult());
        }

        logCollector.collect(cycleId, ClientLoggerMessage.appendMessage(ClientLoggerMessage.INSTANCE_FINISH,
            instanceStatus.getEnDesc()));
        LOGGER.info("cycleId: {} cycle end.", cycleId);

        setHistory(taskMaster.getSerialNum(), cycleStartTime, instanceStatus);

        if (!taskMaster.isKilled()) {
            try {
                taskMaster.clear();
                long delayTime = Long.parseLong(jobInstanceInfo.getTimeExpression()) * 1000;
                cycleStartTime = System.currentTimeMillis() + delayTime;
                Thread.sleep(delayTime);

                cycleId = IdUtil.getUniqueId(jobInstanceInfo.getJobId(),
                    jobInstanceInfo.getJobInstanceId(), taskMaster.aquireSerialNum());
                LOGGER.info("cycleId: {} cycle begin.", cycleId);
                cycleStartTime = System.currentTimeMillis();

                JobInstanceInfo.newBuilder(jobInstanceInfo).setScheduleTime(DateTime.now());
                taskMaster.submitInstance(jobInstanceInfo);
            } catch (Throwable ex) {
                taskMaster.killInstance("killed, because of cycle submit failed.");
                LOGGER.error("cycleId: {} cycle submit failed, need to kill.", cycleId, ex);
            }
        } else {
            taskMaster.aquireSerialNum();
        }
    }

    void init(){
       final String jobIdAndInstanceId = IdUtil.getUniqueIdWithoutTask(jobInstanceInfo.getJobId(), jobInstanceInfo.getJobInstanceId());

        // job instance progress report thread
        new Thread(new Runnable() {
            @Override
            public void run() {
                int intervalTimes = 0;

                try {
                    while (!taskMaster.isKilled()) {
                        Thread.sleep(1000);
                        if (intervalTimes++ > 10) {
                            intervalTimes = 0;
                            WorkerReportJobInstanceProgressRequest request = WorkerReportJobInstanceProgressRequest
                                .newBuilder().setJobId(jobInstanceInfo.getJobId())
                                .setJobInstanceId(jobInstanceInfo.getJobInstanceId())
                                .setProgress(getJobInstanceProgress()).build();
                            ServerDiscoveryFactory.getDiscovery(jobInstanceInfo.getGroupId()).getMapMasterRouter()
                                .tell(request, null);
                        }

                        need2KillSelf();
                    }
                } catch (Throwable e) {
                    LOGGER.error("report status error, jobIdAndInstanceId={}.", jobIdAndInstanceId,
                        e);
                }
            }
        }, "Schedulerx-SecondTaskMaster-report-progress-thread-" + jobIdAndInstanceId).start();
    }

    /**
     * 满足下列任意条件需要自杀
     * 1. 与服务端失联超过30秒
     * 2. 或者网格任务没有可用worker列表
     */
    private void need2KillSelf() {
        if (!taskMaster.isInited()){
            return;
        }

        final String jobIdAndInstanceId = IdUtil.getUniqueIdWithoutTask(jobInstanceInfo.getJobId(), jobInstanceInfo.getJobInstanceId());

        if ( HealthTimeHolder.INSTANCE.isServerHeartbeatHealthTimeout(MISS_SERVER_KILL_TIME)) {
            taskMaster.killInstance("killed, because of worker missed active server.");
            LOGGER.warn("Missed server timeout={}ms, kill jobIdAndInstanceId={}.", HealthTimeHolder.INSTANCE.getServerHeartbeatMsInterval(), jobIdAndInstanceId);
            return;
        }

        if (CollectionUtils.isEmpty(taskMaster.getAliveCheckWorkerSet())
            && CollectionUtils.isEmpty(taskMaster.getJobInstanceInfo().getAllWorkers())) {
            LOGGER.warn("Missed useful worker list, kill jobIdAndInstanceId={}.",
                jobIdAndInstanceId);
            taskMaster.killInstance("killed, because of missed useful worker list.");
            return;
        }
    }

    protected String getJobInstanceProgress() {
        secondProgressDetail.setRunningProgress(taskMaster.getJobInstanceProgress());
        secondProgressDetail.setRunningStartTime(cycleStartTime);
        secondProgressDetail.setRecentProgressHistory(Lists.newArrayList(recentProgressHistory));
        String progressJson = JsonUtil.toJson(secondProgressDetail);
        secondProgressDetail.setRunningProgress(null);
        return progressJson;
    }

    private void setHistory(long serialNum, long loopStartTime, InstanceStatus status){
        if (status == InstanceStatus.SUCCESS) {
            secondProgressDetail.getTodayProgressCounter().incrementSuccess();
        } else {
            secondProgressDetail.getTodayProgressCounter().incrementFailed();
        }

        if(!taskMaster.isKilled()){
            secondProgressDetail.getTodayProgressCounter().incrementRunning();
            secondProgressDetail.getTodayProgressCounter().incrementTotal();
        }

        DateTimeFormatter formatter = DateTimeFormat.forPattern(CommonConstants.DATE_TIME_PATTERN);
        // reset today progress counter
        if(DateTime.now().dayOfMonth().get() != DateTime.parse(secondProgressDetail.getTodayBeginTime(), formatter).dayOfMonth().get()){
            secondProgressDetail.setYesterdayProgressCounter(secondProgressDetail.getTodayProgressCounter());
            secondProgressDetail.setTodayBeginTime(DateTime.now().toString(CommonConstants.DATE_TIME_PATTERN));
            secondProgressDetail.setTodayProgressCounter(new TaskProgressCounter(secondProgressDetail.getTodayBeginTime()));
        }

        Map<String, TaskProgressCounter> taskProgressMap = null;
        String ipAndPort = ActorPathUtil.getIpAndPortFromAkkaPath(taskMaster.getLocalTaskRouterPath());

        if (taskMaster instanceof MapTaskMaster) {
            taskProgressMap = Maps.newHashMap(((MapTaskMaster) taskMaster).getTaskProgressMap());
        } else if(taskMaster instanceof BroadcastTaskMaster){
            Map<String, WorkerProgressCounter> workerProgressCounterMap = ((BroadcastTaskMaster)taskMaster).getWorkerProgressMap();
            if (MapUtils.isEmpty(workerProgressCounterMap)) {
                return;
            }
            TaskProgressCounter counter = new TaskProgressCounter(ipAndPort);
            for (WorkerProgressCounter worker : workerProgressCounterMap.values()) {
                counter.incrementSuccess(worker.getSuccess());
                counter.incrementFailed(worker.getFailed());
                counter.incrementTotal(worker.getTotal());
            }
            taskProgressMap = Maps.newHashMap();
            taskProgressMap.put(ipAndPort, counter);
        } else if(taskMaster instanceof StandaloneTaskMaster){
            taskProgressMap = Maps.newHashMap();
            TaskProgressCounter counter = new TaskProgressCounter(ipAndPort);
            taskProgressMap.put(ipAndPort, counter);
            counter.incrementTotal();
            if (status == InstanceStatus.SUCCESS) {
                counter.incrementSuccess();
            } else {
                counter.incrementFailed();
            }
        }

        if(MapUtils.isEmpty(taskProgressMap)){
            return;
        }

        ProgressHistory history = new ProgressHistory();
        history.setSerialNum(serialNum);
        history.setStartTime(loopStartTime);
        history.setEndTime(System.currentTimeMillis());
        history.setCostTime(history.getEndTime() -  history.getStartTime());
        history.setTaskProgressMap(taskProgressMap);
        history.setSuccess((status == InstanceStatus.SUCCESS) ? true : false);
        recentProgressHistory.offer(history);
    }
}
