package com.alibaba.schedulerx.worker.master;

import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.atomic.AtomicLong;

import com.alibaba.schedulerx.common.domain.InstanceStatus;
import com.alibaba.schedulerx.common.domain.JobInstanceData;
import com.alibaba.schedulerx.common.domain.JobInstanceInfo;
import com.alibaba.schedulerx.common.domain.MapTaskXAttrs;
import com.alibaba.schedulerx.common.domain.TaskStatus;
import com.alibaba.schedulerx.common.util.ConfigUtil;
import com.alibaba.schedulerx.common.util.IdUtil;
import com.alibaba.schedulerx.common.util.JsonUtil;
import com.alibaba.schedulerx.protocol.Common.UpstreamData;
import com.alibaba.schedulerx.protocol.Server.RetryTaskEntity;
import com.alibaba.schedulerx.protocol.Worker.ContainerBatchReportTaskStatuesRequest;
import com.alibaba.schedulerx.protocol.Worker.ContainerReportTaskStatusRequest;
import com.alibaba.schedulerx.protocol.Worker.MasterStartContainerRequest;
import com.alibaba.schedulerx.protocol.Worker.TaskStatusInfo;
import com.alibaba.schedulerx.worker.discovery.ServerDiscovery;
import com.alibaba.schedulerx.worker.discovery.ServerDiscoveryFactory;
import com.alibaba.schedulerx.worker.domain.WorkerConstants;
import com.alibaba.schedulerx.worker.log.LogFactory;
import com.alibaba.schedulerx.worker.log.Logger;
import com.alibaba.schedulerx.worker.master.handler.UpdateInstanceStatusHandler;
import com.alibaba.schedulerx.worker.master.handler.UpdateInstanceStatusHandlerFactory;
import com.alibaba.schedulerx.worker.processor.ProcessResult;

import akka.actor.ActorContext;
import com.google.common.collect.Maps;
import com.google.common.collect.Sets;
import com.google.protobuf.ByteString;

/**
 * @author xiaomeng.hxm
 */
public abstract class TaskMaster {
    private final ActorContext actorContext;
    protected volatile InstanceStatus instanceStatus = InstanceStatus.RUNNING;
    protected Map<String, TaskStatus> taskStatusMap = Maps.newHashMap();
    protected AtomicLong taskIdGenerator = new AtomicLong(0);
    //private final String localWorkerAkkaPath;
    private final String localContainerRouterPath;
    private final String localTaskRouterPath;
    private final String localInstanceRouterPath;
    protected final JobInstanceInfo jobInstanceInfo;
    protected final UpdateInstanceStatusHandler statusHandler;
    protected volatile boolean killed = false;
    protected volatile boolean INITED = false;
    protected Set<String> aliveCheckWorkerSet = Sets.newConcurrentHashSet();
    protected final ServerDiscovery SERVER_DISCOVERY;
    // 秒级任务使用，当前循环次数
    protected AtomicLong serialNum = new AtomicLong(0);
    private static final Logger LOGGER = LogFactory.getLogger(TaskMaster.class);

    public TaskMaster(JobInstanceInfo jobInstanceInfo, ActorContext actorContext) throws Exception {
        this.jobInstanceInfo = jobInstanceInfo;
        this.actorContext = actorContext;
        this.localInstanceRouterPath = actorContext.provider().getDefaultAddress().toString()
                + WorkerConstants.WORKER_AKKA_JOB_INSTANCE_ROUTING_PATH;
        this.localContainerRouterPath = actorContext.provider().getDefaultAddress().toString()
            + WorkerConstants.WORKER_AKKA_CONTAINER_ROUTING_PATH;
        this.localTaskRouterPath = actorContext.provider().getDefaultAddress().toString()
            + WorkerConstants.WORKER_AKKA_TASK_ROUTING_PATH;
        this.SERVER_DISCOVERY = ServerDiscoveryFactory.getDiscovery(jobInstanceInfo.getGroupId());
        this.aliveCheckWorkerSet.addAll(jobInstanceInfo.getAllWorkers());
        checkProcessor();
        this.statusHandler = UpdateInstanceStatusHandlerFactory.create(this, jobInstanceInfo);
    }

    public ActorContext getActorContext() {
        return actorContext;
    }

    public String getLocalJobInstanceRouterPath() {
        return localInstanceRouterPath;
    }

    public String getLocalContainerRouterPath() {
        return localContainerRouterPath;
    }

    public String getLocalTaskRouterPath() {
        return localTaskRouterPath;
    }

    public boolean isJobInstanceFinished() {
        boolean isFinish = true;
        //TODO 需要重构，效率低下
        for (TaskStatus status : taskStatusMap.values()) {
            if (!status.isFinish()) {
                isFinish = false;
                break;
            }
        }
        return isFinish;
    }

    public void updateTaskStatus(ContainerReportTaskStatusRequest request) throws Exception {
        long jobId = request.getJobId();
        long jobInstanceId = request.getJobInstanceId();
        long taskId = request.getTaskId();
        TaskStatus taskStatus = TaskStatus.parseValue(request.getStatus());

        //TODO 这段代码写得特别挫，效率极其低下，需要重构
        String uniqueId = IdUtil.getUniqueId(jobId, jobInstanceId, taskId);
        taskStatusMap.put(uniqueId, taskStatus);

        InstanceStatus newStatus = InstanceStatus.UNKNOWN;
        if (taskStatusMap.size() > 0) {
            if (!isJobInstanceFinished()) {
                newStatus = InstanceStatus.RUNNING;
            } else {
                newStatus = InstanceStatus.SUCCESS;
                //只要有一个子任务状态为FAILED，则返回FAILED
                if (!newStatus.equals(InstanceStatus.FAILED)) {
                    for (TaskStatus status : taskStatusMap.values()) {
                        if (status.equals(TaskStatus.FAILED)) {
                            newStatus = InstanceStatus.FAILED;
                            break;
                        }
                    }
                }
            }
        }

        updateNewInstanceStatus(request.getSerialNum(), jobInstanceId, newStatus, request.getResult());
    }

    //TODO: MapTaskMaster may override this method do really batch process
    public void batchUpdateTaskStatus(ContainerBatchReportTaskStatuesRequest request) throws Exception {
        for (TaskStatusInfo taskStatusInfo : request.getTaskStatuesList()) {
            ContainerReportTaskStatusRequest.Builder builder = ContainerReportTaskStatusRequest.newBuilder()
                .setJobId(request.getJobId())
                .setJobInstanceId(request.getJobInstanceId())
                .setTaskId(taskStatusInfo.getTaskId())
                .setWorkerAddr(request.getWorkerAddr())
                .setWorkerId(request.getWorkerId())
                .setStatus(taskStatusInfo.getStatus());
            if (taskStatusInfo.hasResult()) {
                builder.setResult(taskStatusInfo.getResult());
            }
            if (taskStatusInfo.hasTaskName()) {
                builder.setTaskName(taskStatusInfo.getTaskName());
            }
            if (request.hasSerialNum()) {
                builder.setSerialNum(request.getSerialNum());
            }
            updateTaskStatus(builder.build());
        }
    }

    public void killInstance(String reason) {
        this.killed = true;
    }

    public abstract void destroyContainerPool();

    public void killTask(String uniqueId, String workerId, String workerAddr) {
        //TODO
    }

    protected void init() {
        if (INITED) {
            return;
        }
        INITED = true;
//        new Thread(new Runnable() {
//            @Override
//            public void run() {
//                while (!instanceStatus.isFinish()) {
//                    WorkerReportJobInstanceProgressRequest request = WorkerReportJobInstanceProgressRequest.newBuilder()
//                        .setJobId(jobInstanceInfo.getJobId())
//                        .setJobInstanceId(jobInstanceInfo.getJobInstanceId())
//                        .setProgress(getJobInstanceProgress())
//                        .build();
//                    SERVER_DISCOVERY.getInstanceStatusRouter().tell(request, null);
//                    LOGGER.debug("update progress, jobId={}, jobInstanceId={}", jobInstanceInfo.getJobId(), jobInstanceInfo.getJobInstanceId());
//                    try {
//                        //每隔10秒更新进度
//                        Thread.sleep(10000);
//                    } catch (InterruptedException e) {
//                        LOGGER.error("", e);
//                        break;
//                    }
//                }
//            }
//        }, "Schedulerx-TaskMaster-report-progress-thread-" + jobInstanceInfo.getJobId() + "_" + jobInstanceInfo
//            .getJobInstanceId()).start();
//
//        LOGGER.debug("Schedulerx-TaskMaster-report-progress-thread-" + jobInstanceInfo.getJobId() + "_" + jobInstanceInfo
//            .getJobInstanceId() + " started.");
    }

    public void retryTasks(List<RetryTaskEntity> taskEntities) {}

    public abstract void submitInstance(JobInstanceInfo jobInstanceInfo) throws Exception;

    protected long aquireTaskId() {
        return taskIdGenerator.getAndIncrement();
    }

    public String getJobInstanceProgress() {
        return "0.5";
    }

    public void updateNewInstanceStatus(long serialNum, InstanceStatus newStatus, String result) {
        updateNewInstanceStatus(serialNum, jobInstanceInfo.getJobInstanceId(), newStatus, result);
    }

    protected synchronized void updateNewInstanceStatus(long serialNum, long jobInstanceId, InstanceStatus newStatus,
        String result) {
        try {
            this.statusHandler.handle(serialNum, newStatus, result);
        } catch (Exception e) {
            LOGGER.error("update jobInstanceId={} serialNum={}, status={} failed", jobInstanceId, serialNum, newStatus.getValue(), e);
        }
    }

    public void stop() {

    }

    public void clear() {
        taskStatusMap.clear();
        taskIdGenerator.set(0);
        instanceStatus = InstanceStatus.RUNNING;
        aliveCheckWorkerSet.clear();
        boolean enableShareContainerPool = ConfigUtil.getWorkerConfig().getBoolean(WorkerConstants.SHARE_CONTAINER_POOL, false);
        if (!enableShareContainerPool) {
            destroyContainerPool();
        }
    }

    public ProcessResult postFinish(long jobInstanceId) {
        return null;
    }

    protected MasterStartContainerRequest convert2StartContainerRequest(JobInstanceInfo jobInstanceInfo, long taskId) {
        return convert2StartContainerRequest(jobInstanceInfo, taskId, null, null);
    }
    
    protected MasterStartContainerRequest convert2StartContainerRequest(JobInstanceInfo jobInstanceInfo, long taskId,
            String taskName, ByteString taskBody) {
        return convert2StartContainerRequest(jobInstanceInfo, taskId, taskName, taskBody, false);
    }
    
    protected MasterStartContainerRequest convert2StartContainerRequest(JobInstanceInfo jobInstanceInfo, long taskId,
            String taskName, ByteString taskBody, boolean failover) {
        MasterStartContainerRequest.Builder builder = MasterStartContainerRequest.newBuilder();
        builder.setJobId(jobInstanceInfo.getJobId());
        builder.setJobInstanceId(jobInstanceInfo.getJobInstanceId());
        builder.setTaskId(taskId);
        builder.setUser(jobInstanceInfo.getUser());
        builder.setJobType(jobInstanceInfo.getJobType());
        builder.setContent(jobInstanceInfo.getContent());
        builder.setScheduleTime(jobInstanceInfo.getScheduleTime().getMillis());
        builder.setDataTime(jobInstanceInfo.getDataTime().getMillis());
        builder.setParameters(jobInstanceInfo.getParameters());
        builder.setInstanceParameters(jobInstanceInfo.getInstanceParameters());
        builder.setInstanceMasterAkkaPath(getLocalTaskRouterPath());
        builder.setGroupId(jobInstanceInfo.getGroupId());
        builder.setMaxAttempt(jobInstanceInfo.getMaxAttempt());
        builder.setAttempt(jobInstanceInfo.getAttempt());
        if (jobInstanceInfo.getUpstreamData() != null && !jobInstanceInfo.getUpstreamData().isEmpty()) {
            for (JobInstanceData jobInstanceData : jobInstanceInfo.getUpstreamData()) {
                UpstreamData upstreamData = UpstreamData.newBuilder()
                        .setJobName(jobInstanceData.getJobName())
                        .setData(jobInstanceData.getData())
                        .build();
                builder.addUpstreamData(upstreamData);
            }
        }
        if (jobInstanceInfo.getXattrs() != null && !jobInstanceInfo.getXattrs().isEmpty()) {
            MapTaskXAttrs xAttrs = JsonUtil.fromJson(jobInstanceInfo.getXattrs(), MapTaskXAttrs.class);
            builder.setConsumerNum(xAttrs.getConsumerSize());
            builder.setTaskMaxAttempt(xAttrs.getTaskMaxAttempt());
            builder.setTaskAttemptInterval(xAttrs.getTaskAttemptInterval());
        }
        if (taskName != null) {
            builder.setTaskName(taskName);
        }
        if (taskBody != null) {
            builder.setTask(taskBody);
        }
        if (failover) {
            builder.setFailover(true);
        }
        if (jobInstanceInfo.getWfInstanceId() != null) {
            builder.setWfInstanceId(jobInstanceInfo.getWfInstanceId());
        }
        builder.setSerialNum(getSerialNum());
        builder.setExecuteMode(jobInstanceInfo.getExecuteMode());
        if (jobInstanceInfo.getJobName() != null) {
            builder.setJobName(jobInstanceInfo.getJobName());
        }

        return builder.build();
    }

    /**
     * Getter method for property <tt>instanceStatus</tt>.
     *
     * @return property value of instanceStatus
     */
    public InstanceStatus getInstanceStatus() {
        return instanceStatus;
    }

    /**
     * Setter method for property <tt>instanceStatus </tt>.
     *
     * @param instanceStatus value to be assigned to property instanceStatus
     */
    public void setInstanceStatus(InstanceStatus instanceStatus) {
        this.instanceStatus = instanceStatus;
    }

    public boolean isKilled() {
        return killed;
    }

    /**
     * Getter method for property <tt>jobInstanceInfo</tt>.
     *
     * @return property value of jobInstanceInfo
     */
    public JobInstanceInfo getJobInstanceInfo() {
        return jobInstanceInfo;
    }

    /**
     * Getter method for property <tt>aliveCheckWorkerSet</tt>.
     *
     * @return property value of aliveCheckWorkerSet
     */
    public Set<String> getAliveCheckWorkerSet() {
        return aliveCheckWorkerSet;
    }

    public boolean isInited(){
        return INITED;
    }

    public long getSerialNum() {
        return serialNum.get();
    }

    public long aquireSerialNum() {
        return serialNum.incrementAndGet();
    }

    protected void checkProcessor() throws Exception {}
}
