package com.alibaba.schedulerx.worker.master;

import java.io.IOException;

import com.alibaba.schedulerx.common.domain.InstanceStatus;
import com.alibaba.schedulerx.common.domain.JobInstanceInfo;
import com.alibaba.schedulerx.common.domain.TaskStatus;
import com.alibaba.schedulerx.common.util.IdUtil;
import com.alibaba.schedulerx.protocol.Worker.MasterDestroyContainerPoolRequest;
import com.alibaba.schedulerx.protocol.Worker.MasterKillContainerRequest;
import com.alibaba.schedulerx.protocol.Worker.MasterKillContainerResponse;
import com.alibaba.schedulerx.protocol.Worker.MasterStartContainerRequest;
import com.alibaba.schedulerx.protocol.Worker.MasterStartContainerResponse;
import com.alibaba.schedulerx.protocol.utils.FutureUtils;
import com.alibaba.schedulerx.worker.log.LogFactory;
import com.alibaba.schedulerx.worker.log.Logger;
import com.alibaba.schedulerx.worker.logcollector.ClientLoggerMessage;
import com.alibaba.schedulerx.worker.logcollector.LogCollector;
import com.alibaba.schedulerx.worker.logcollector.LogCollectorFactory;
import com.alibaba.schedulerx.worker.processor.JobProcessor;
import com.alibaba.schedulerx.worker.processor.MapJobProcessor;
import com.alibaba.schedulerx.worker.util.JavaProcessorProfileUtil;

import akka.actor.ActorContext;
import akka.actor.ActorSelection;


/**
 *
 * @author xiaomeng.hxm
 */
public class StandaloneTaskMaster extends TaskMaster {

    private LogCollector logCollector = LogCollectorFactory.get();
    private static final Logger LOGGER = LogFactory.getLogger(StandaloneTaskMaster.class);

    public StandaloneTaskMaster(JobInstanceInfo jobInstanceInfo, ActorContext actorContext) throws Exception {
        super(jobInstanceInfo, actorContext);
    }

    /**
     * Standalone TaskMaster only submit to localhost
     */
    @Override
    public void submitInstance(JobInstanceInfo info) {
        ActorSelection selection = getActorContext().actorSelection(getLocalContainerRouterPath());
        long taskId = aquireTaskId();
        String uniqueId = IdUtil.getUniqueId(info.getJobId(), info.getJobInstanceId(), taskId);
        MasterStartContainerRequest request = convert2StartContainerRequest(info, taskId);
        try {
            MasterStartContainerResponse response = (MasterStartContainerResponse) FutureUtils.awaitResult(selection, request, 10);
            if (response.getSuccess()) {
                taskStatusMap.put(uniqueId, TaskStatus.INIT);
                logCollector.collect(uniqueId, ClientLoggerMessage.STANDALONE_INSTANCE_INIT_SUCCESS);
            } else {
                LOGGER.error("submitTask error.", response.getMessage());
                taskStatusMap.put(uniqueId, TaskStatus.FAILED);
                logCollector.collect(uniqueId, ClientLoggerMessage.appendMessage(ClientLoggerMessage.STANDALONE_INSTANCE_INIT_FAIL,
                        response.getMessage()));
            }

            init();
        } catch (Throwable e) {
            LOGGER.error("submitTask error.", e);
            logCollector.collect(uniqueId, ClientLoggerMessage.STANDALONE_INSTANCE_INIT_FAIL, e);
            taskStatusMap.put(uniqueId, TaskStatus.FAILED);
        }
    }

    @Override
    public void killInstance(String reason) {
        super.killInstance(reason);
        updateNewInstanceStatus(getSerialNum(), jobInstanceInfo.getJobInstanceId(), InstanceStatus.FAILED,
            reason);
        MasterKillContainerResponse response = null;
        ActorSelection selection = getActorContext().actorSelection(getLocalContainerRouterPath());
        String uniqueId = IdUtil.getUniqueIdWithoutTask(jobInstanceInfo.getJobId(), jobInstanceInfo.getJobInstanceId());
        if(!instanceStatus.isFinish()){
            instanceStatus = InstanceStatus.FAILED;
        }
        try {
            MasterKillContainerRequest request = MasterKillContainerRequest.newBuilder()
                                                                           .setJobId(jobInstanceInfo.getJobId())
                                                                           .setJobInstanceId(jobInstanceInfo.getJobInstanceId())
                                                                           .build();
            response = (MasterKillContainerResponse)FutureUtils.awaitResult(selection, request, 10);
            if (response.getSuccess()) {
                LOGGER.info("[StandaloneTaskMaster]kill standalone instance success, uniqueId:{}",uniqueId);
            } else {
                LOGGER.warn("[StandaloneTaskMaster]kill standalone instance failed, uniqueId:{}",uniqueId);
            }
        } catch (Throwable e) {
            LOGGER.error("[StandaloneTaskMaster]kill standalone instance exception uniqueId:{}",uniqueId, e);
        }
    }

    @Override
    public void destroyContainerPool() {
        try {
            ActorSelection selection = getActorContext().actorSelection(getLocalContainerRouterPath());
            MasterDestroyContainerPoolRequest request = MasterDestroyContainerPoolRequest.newBuilder()
                .setJobInstanceId(jobInstanceInfo.getJobInstanceId())
                .setSerialNum(getSerialNum())
                .build();
            FutureUtils.awaitResult(selection, request, 5);
        } catch (Throwable e) {
            LOGGER.error("destroy containerPool failed", e);
        }

    }

    @Override
    protected void checkProcessor() throws Exception {
        if ("java".equalsIgnoreCase(jobInstanceInfo.getJobType())) {
            JobProcessor processor = JavaProcessorProfileUtil.getJavaProcessor(jobInstanceInfo.getContent());
            if (processor instanceof MapJobProcessor) {
                throw new IOException(processor.getClass().getName() + " shouldn't extends MapJobProcessor or MapReduceJobProcessor");
            }
        }
    }
}
