package com.alibaba.schedulerx.worker.master;

import java.io.IOException;
import java.util.List;
import java.util.Map;
import java.util.concurrent.TimeoutException;

import com.alibaba.schedulerx.common.domain.InstanceStatus;
import com.alibaba.schedulerx.common.domain.JobInstanceInfo;
import com.alibaba.schedulerx.common.domain.MapTaskProgress;
import com.alibaba.schedulerx.common.domain.TaskStatus;
import com.alibaba.schedulerx.common.domain.WorkerProgressCounter;
import com.alibaba.schedulerx.common.util.ExceptionUtil;
import com.alibaba.schedulerx.common.util.IdUtil;
import com.alibaba.schedulerx.common.util.JsonUtil;
import com.alibaba.schedulerx.protocol.Worker.ContainerReportTaskStatusRequest;
import com.alibaba.schedulerx.protocol.Worker.MasterCheckWorkerAliveRequest;
import com.alibaba.schedulerx.protocol.Worker.MasterDestroyContainerPoolRequest;
import com.alibaba.schedulerx.protocol.Worker.MasterKillContainerRequest;
import com.alibaba.schedulerx.protocol.Worker.MasterStartContainerRequest;
import com.alibaba.schedulerx.protocol.Worker.MasterStartContainerResponse;
import com.alibaba.schedulerx.protocol.utils.FutureUtils;
import com.alibaba.schedulerx.worker.SchedulerxWorker;
import com.alibaba.schedulerx.worker.domain.JobContext;
import com.alibaba.schedulerx.worker.log.LogFactory;
import com.alibaba.schedulerx.worker.log.Logger;
import com.alibaba.schedulerx.worker.logcollector.ClientLoggerMessage;
import com.alibaba.schedulerx.worker.logcollector.LogCollector;
import com.alibaba.schedulerx.worker.logcollector.LogCollectorFactory;
import com.alibaba.schedulerx.worker.processor.JobProcessor;
import com.alibaba.schedulerx.worker.processor.MapJobProcessor;
import com.alibaba.schedulerx.worker.processor.ProcessResult;
import com.alibaba.schedulerx.worker.util.ActorPathUtil;
import com.alibaba.schedulerx.worker.util.JavaProcessorProfileUtil;
import com.alibaba.schedulerx.worker.util.WorkerIdGenerator;

import akka.actor.ActorContext;
import akka.actor.ActorSelection;
import com.google.common.collect.Maps;

/**
 * @author xiaomeng.hxm
 */
public class BroadcastTaskMaster extends TaskMaster {

    private static final Logger LOGGER = LogFactory.getLogger(BroadcastTaskMaster.class);
    private Map<String, String> worker2uniqueIdMap = Maps.newConcurrentMap();
    private Map<String, WorkerProgressCounter> workerProgressMap = Maps.newConcurrentMap();
    private LogCollector logCollector = LogCollectorFactory.get();

    public BroadcastTaskMaster(JobInstanceInfo jobInstanceInfo, ActorContext actorContext) throws Exception {
        super(jobInstanceInfo, actorContext);
    }

    @Override
    public void submitInstance(JobInstanceInfo info) {
        if ("java".equalsIgnoreCase(info.getJobType())) {
            try {
                preProcess(info);
            } catch (Exception e) {
                LOGGER.error("BroadcastTaskMaster.preProcess failed, jobInstanceId={}", info.getJobInstanceId(), e);
                String uniqueId = IdUtil.getUniqueId(info.getJobId(), info.getJobInstanceId(), 0);
                logCollector.collect(uniqueId, ClientLoggerMessage.appendMessage(ClientLoggerMessage.BROADCAST_INSTANCE_INIT_FAIL,
                        SchedulerxWorker.WORKER_ADDR, ExceptionUtil.getMessage(e)));
                ContainerReportTaskStatusRequest faileRequest = ContainerReportTaskStatusRequest.newBuilder()
                        .setJobId(info.getJobId())
                        .setJobInstanceId(info.getJobInstanceId())
                        .setTaskId(0)
                        .setStatus(TaskStatus.FAILED.getValue())
                        .setWorkerId(WorkerIdGenerator.get())
                        .setWorkerAddr(SchedulerxWorker.WORKER_ADDR)
                        .build();
                updateTaskStatus(faileRequest);
                return;
            }
        }

        List<String> allWorkers = info.getAllWorkers();
        for (String workerIdAddr : allWorkers) {
            String[] workerInfo = workerIdAddr.split("@");
            String workerAddr = workerInfo[1];
            String workerId = workerInfo[0];
            ActorSelection selection = getActorContext().actorSelection(ActorPathUtil.getContainerRouterPath(workerIdAddr));
            long taskId = aquireTaskId();
            String uniqueId = IdUtil.getUniqueId(info.getJobId(), info.getJobInstanceId(), taskId);
            MasterStartContainerRequest request = convert2StartContainerRequest(info, taskId);
            try {
                taskStatusMap.put(uniqueId, TaskStatus.RUNNING);
                if (!workerProgressMap.containsKey(workerAddr)) {
                    WorkerProgressCounter workerProgressCounter = new WorkerProgressCounter(workerAddr);
                    workerProgressMap.put(workerAddr, workerProgressCounter);
                }
                workerProgressMap.get(workerAddr).incrementTotal();
                workerProgressMap.get(workerAddr).incrementRunning();

                MasterStartContainerResponse response = (MasterStartContainerResponse)FutureUtils.awaitResult(selection,
                    request, 10);
                if (response.getSuccess()) {
                    worker2uniqueIdMap.put(workerIdAddr, uniqueId);
                    logCollector.collect(uniqueId, ClientLoggerMessage
                            .appendMessage(ClientLoggerMessage.BROADCAST_INSTANCE_INIT_SUCCESS, workerAddr));
                } else {
                    LOGGER.error("submitTask[{}] to worker error, {}", uniqueId, workerAddr, response.getMessage());
                    logCollector.collect(uniqueId, ClientLoggerMessage.appendMessage(ClientLoggerMessage.BROADCAST_INSTANCE_INIT_FAIL,
                            workerAddr, response.getMessage()));
                    ContainerReportTaskStatusRequest faileRequest = ContainerReportTaskStatusRequest.newBuilder()
                            .setJobId(info.getJobId())
                            .setJobInstanceId(info.getJobInstanceId())
                            .setTaskId(taskId)
                            .setStatus(TaskStatus.FAILED.getValue())
                            .setWorkerId(workerId)
                            .setWorkerAddr(workerAddr)
                            .build();
                    updateTaskStatus(faileRequest);
                }
            } catch (Throwable e) {
                LOGGER.error("start container failed, worker:{}, uniqueId:{}", workerAddr,
                    uniqueId, e);
                logCollector.collect(uniqueId, ClientLoggerMessage
                        .appendMessage(ClientLoggerMessage.BROADCAST_INSTANCE_INIT_FAIL, workerAddr), e);
                ContainerReportTaskStatusRequest faileRequest = ContainerReportTaskStatusRequest.newBuilder()
                        .setJobId(info.getJobId())
                        .setJobInstanceId(info.getJobInstanceId())
                        .setTaskId(taskId)
                        .setStatus(TaskStatus.FAILED.getValue())
                        .setWorkerAddr(workerAddr)
                        .setWorkerId(workerId)
                        .build();
                updateTaskStatus(faileRequest);
            }
        }

        init();
    }

    @Override
    public void killInstance(String reason) {
        super.killInstance(reason);
        String uniqueId = IdUtil.getUniqueIdWithoutTask(jobInstanceInfo.getJobId(), jobInstanceInfo.getJobInstanceId());
        updateNewInstanceStatus(getSerialNum(), jobInstanceInfo.getJobInstanceId(), InstanceStatus.FAILED,
            reason);
        List<String> allWorkers = jobInstanceInfo.getAllWorkers();
        for (String workerIdAddr : allWorkers) {
            try {
                ActorSelection selection = getActorContext().actorSelection(ActorPathUtil.getContainerRouterPath(workerIdAddr));
                MasterKillContainerRequest request = MasterKillContainerRequest.newBuilder()
                    .setJobId(jobInstanceInfo.getJobId())
                    .setJobInstanceId(jobInstanceInfo.getJobInstanceId())
                    .build();
                selection.tell(request, null);
            } catch (Throwable e) {
                logCollector.collect(uniqueId,
                        ClientLoggerMessage.appendMessage(ClientLoggerMessage.INSTANCE_KILL_FAIL, workerIdAddr), e);
                LOGGER.error("send kill instance request exception, worker:{}, uninqueId:{}",
                        workerIdAddr, uniqueId);
            }
        }
    }

    @Override
    public void destroyContainerPool() {
        List<String> allWorkers = jobInstanceInfo.getAllWorkers();
        for (String workerIdAddr : allWorkers) {
            MasterDestroyContainerPoolRequest request = MasterDestroyContainerPoolRequest.newBuilder()
                .setJobInstanceId(jobInstanceInfo.getJobInstanceId())
                .setJobId(jobInstanceInfo.getJobId())
                .setWorkerIdAddr(workerIdAddr)
                .setSerialNum(getSerialNum())
                .build();
            SchedulerxWorker.AtLeastDeliveryRoutingActor.tell(request, null);
        }
    }

    @Override
    public synchronized void updateTaskStatus(ContainerReportTaskStatusRequest request) {
        long jobId = request.getJobId();
        long jobInstanceId = request.getJobInstanceId();
        long taskId = request.getTaskId();
        String workerAddr = request.getWorkerAddr();
        TaskStatus taskStatus = TaskStatus.parseValue(request.getStatus());
        
        String uniqueId = IdUtil.getUniqueId(jobId, jobInstanceId, taskId);
        taskStatusMap.put(uniqueId, taskStatus);
        LOGGER.info("update task status, uniqueId={}, status={}, workerAddr={}", uniqueId, 
                taskStatus.getDescription(), workerAddr);

        if (!workerProgressMap.containsKey(workerAddr)) {
            WorkerProgressCounter workerProgressCounter = new WorkerProgressCounter(workerAddr);
            workerProgressMap.put(workerAddr, workerProgressCounter);
        }
        if (taskStatus.equals(TaskStatus.RUNNING)) {
            workerProgressMap.get(workerAddr).incrementRunning();
        } else if (taskStatus.equals(TaskStatus.SUCCESS)) {
            workerProgressMap.get(workerAddr).incrementSuccess();
        } else if (taskStatus.equals(TaskStatus.FAILED)) {
            workerProgressMap.get(workerAddr).incrementFailed();
        }

        InstanceStatus newStatus = InstanceStatus.UNKNOWN;
        if (taskStatusMap.size() > 0) {
            if (!isJobInstanceFinished()) {
                newStatus = InstanceStatus.RUNNING;
            } else {
                newStatus = InstanceStatus.SUCCESS;
                //只要有一个子任务状态为FAILED，则返回FAILED
                if (!newStatus.equals(InstanceStatus.FAILED)) {
                    for (TaskStatus status : taskStatusMap.values()) {
                        if (status.equals(TaskStatus.FAILED)) {
                            newStatus = InstanceStatus.FAILED;
                            break;
                        }
                    }
                }
            }
        }
        updateNewInstanceStatus(request.getSerialNum(), jobInstanceId, newStatus, request.getResult());
        LOGGER.info("update jobInstance status={}", newStatus.getDescription());
    }

    @Override
    public String getJobInstanceProgress() {
        MapTaskProgress detail = new MapTaskProgress();
        detail.setWorkerProgress(workerProgressMap.values());
        return JsonUtil.toJson(detail);
    }

    @Override
    protected void init() {
        if(INITED){
            return;
        }

        super.init();

        new Thread(new Runnable() {
            @Override
            public void run() {
                while (!instanceStatus.isFinish()) {
                    aliveCheckWorkerSet.addAll(jobInstanceInfo.getAllWorkers());
                    for (String workerIdAddr : aliveCheckWorkerSet) {
                        try {
                            ActorSelection selection = getActorContext().actorSelection(
                                ActorPathUtil.getWorkerHeartbeatRouterPath(workerIdAddr)
                            );
                            MasterCheckWorkerAliveRequest request = MasterCheckWorkerAliveRequest.newBuilder()
                                .setJobInstanceId(jobInstanceInfo.getJobInstanceId())
                                .build();
                            FutureUtils.awaitResult(selection, request, 10);
                        } catch (TimeoutException e) {
                            String uniqueId = worker2uniqueIdMap.get(workerIdAddr);
                            if (uniqueId != null) {
                                String[] workerInfo = workerIdAddr.split("@");
                                String workerAddr = workerInfo[1];
                                String workerId = workerInfo[0];
                                String[] tokens = uniqueId.split(IdUtil.SPLITTER_TOKEN);
                                ContainerReportTaskStatusRequest request = ContainerReportTaskStatusRequest.newBuilder()
                                    .setJobId(Long.valueOf(tokens[0]))
                                    .setJobInstanceId(Long.valueOf(tokens[1]))
                                    .setTaskId(Long.valueOf(tokens[2]))
                                    .setStatus(TaskStatus.FAILED.getValue())
                                    .setWorkerAddr(workerAddr)
                                    .setWorkerId(workerId)
                                    .build();
                                updateTaskStatus(request);
                                LOGGER.warn("worker[{}] is down, set {} to failed", workerAddr, uniqueId);
                            } else {
                                LOGGER.error("can't found workerAddr of uniqueId={}", uniqueId);
                            }
                        } catch (Throwable e) {
                            LOGGER.error("", e);
                        }
                    }
                    try {
                        Thread.sleep(5000);
                    } catch (InterruptedException e) {
                        LOGGER.error("", e);
                        break;
                    }
                }
            }
        }, "Schedulerx-BroadcastTaskMaster-check-worker-alive-thread-" + jobInstanceInfo.getJobId() + "_"
            + jobInstanceInfo.getJobInstanceId()).start();

    }

    /**
     * Getter method for property <tt>workerProgressMap</tt>.
     *
     * @return property value of workerProgressMap
     */
    public Map<String, WorkerProgressCounter> getWorkerProgressMap() {
        return workerProgressMap;
    }

    @Override
    protected void checkProcessor() throws Exception {
        if ("java".equalsIgnoreCase(jobInstanceInfo.getJobType())) {
            JobProcessor processor = JavaProcessorProfileUtil.getJavaProcessor(jobInstanceInfo.getContent());
            if (processor instanceof MapJobProcessor) {
                throw new IOException(processor.getClass().getName() + " shouldn't extends MapJobProcessor or MapReduceJobProcessor");
            }
        }
    }

    @SuppressWarnings("resource")
    @Override
    public ProcessResult postFinish(long jobInstanceId) {
        ProcessResult postResult = null;
        if ("java".equalsIgnoreCase(jobInstanceInfo.getJobType())) {
            try {
                JobContext context = JobContext.newBuilder()
                        .setJobId(jobInstanceInfo.getJobId())
                        .setJobInstanceId(jobInstanceId)
                        .setJobType(jobInstanceInfo.getJobType())
                        .setContent(jobInstanceInfo.getContent())
                        .setScheduleTime(jobInstanceInfo.getScheduleTime())
                        .setDataTime(jobInstanceInfo.getDataTime())
                        .setJobParameters(jobInstanceInfo.getParameters())
                        .setInstanceParameters(jobInstanceInfo.getInstanceParameters())
                        .setUser(jobInstanceInfo.getUser())
                        .build();
                JobProcessor jobProcessor = JavaProcessorProfileUtil.getJavaProcessor(context.getContent());
                postResult = jobProcessor.postProcess(context);
            } catch (Throwable e) {
                LOGGER.error("", e);
            }
        }
        return postResult;
    }

    private void preProcess(JobInstanceInfo jobInstanceInfo) throws Exception {
        JobContext context = JobContext.newBuilder()
                .setJobId(jobInstanceInfo.getJobId())
                .setJobInstanceId(jobInstanceInfo.getJobInstanceId())
                .setJobType(jobInstanceInfo.getJobType())
                .setContent(jobInstanceInfo.getContent())
                .setScheduleTime(jobInstanceInfo.getScheduleTime())
                .setDataTime(jobInstanceInfo.getDataTime())
                .setJobParameters(jobInstanceInfo.getParameters())
                .setInstanceParameters(jobInstanceInfo.getInstanceParameters())
                .setUser(jobInstanceInfo.getUser())
                .build();
        JobProcessor jobProcessor = JavaProcessorProfileUtil.getJavaProcessor(context.getContent());
        jobProcessor.preProcess(context);
    }

    @Override
    public void clear() {
        super.clear();
        worker2uniqueIdMap.clear();
        workerProgressMap.clear();
    }
}
