package com.alibaba.schedulerx.worker.discovery;

import java.util.List;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.ScheduledThreadPoolExecutor;
import java.util.concurrent.ThreadPoolExecutor;
import java.util.concurrent.TimeUnit;

import com.alibaba.schedulerx.common.domain.Constants;
import com.alibaba.schedulerx.common.domain.JSONResult;
import com.alibaba.schedulerx.common.domain.ResponseCode;
import com.alibaba.schedulerx.common.domain.ScaleGroupResult;
import com.alibaba.schedulerx.common.util.ConfigUtil;
import com.alibaba.schedulerx.common.util.JsonUtil;
import com.alibaba.schedulerx.worker.SchedulerxWorker;
import com.alibaba.schedulerx.worker.domain.WorkerConstants;
import com.alibaba.schedulerx.worker.log.LogFactory;
import com.alibaba.schedulerx.worker.log.Logger;

import akka.actor.ActorSelection;
import akka.actor.ActorSystem;
import com.google.common.util.concurrent.ThreadFactoryBuilder;
import com.mashape.unirest.http.HttpResponse;
import com.mashape.unirest.http.JsonNode;
import com.mashape.unirest.http.Unirest;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang.StringUtils;

/**
 * heart beat get active leader of specific group through console.
 *
 * @author @yanxun
 * @date 2018/5/22
 */
public class DefaultServerDiscovery implements ServerDiscovery {
    public DefaultServerDiscovery() {}

    private static final Logger LOGGER = LogFactory.getLogger(DefaultServerDiscovery.class);
    private static final String ACTIVE_SERVER_QUERY_PATH = "/worker/v1/appgroup/getLeaderAddr";
    private static final String SERVER_DISCOVERY_THREAD_NAME = "activeServerDiscoveryThread-";
    private ScheduledExecutorService scheduledExecutorService;
    private volatile String activeServerAddr;
    private volatile ActorSelection instanceStatusRouter;
    private volatile ActorSelection mapMasterRouter;
    private volatile ActorSelection taskStatusRouter;
    private volatile ActorSelection heartbeatActor;
    @Deprecated
    private volatile List<ActorSelection> standbyServerHeatbeatActors;
    private volatile ActorSystem actorSystem = SchedulerxWorker.actorSystem;
    private GroupManager groupManager = GroupManager.INSTANCE;

    @Override
    public void start(final String namespace, final String namespaceSource, final String groupId) throws Exception {
        final String consoleDomain = ConfigUtil.getWorkerConfig().getString(WorkerConstants.WORKER_DOMAIN_NAME);
        scheduledExecutorService = new ScheduledThreadPoolExecutor(1,
            new ThreadFactoryBuilder().setNameFormat(SERVER_DISCOVERY_THREAD_NAME + groupId).build(),
            new ThreadPoolExecutor.DiscardPolicy());

        scheduledExecutorService.scheduleAtFixedRate(new Runnable() {
            @Override
            public void run() {
                try {
                    String activeServer = queryActiveServer(consoleDomain, groupId, namespace, namespaceSource);
                    if (!StringUtils.isEmpty(activeServer) && !activeServer.equalsIgnoreCase(activeServerAddr)) {
                        LOGGER.info("activeServerAddr={} change to {}, actorSystem={}", activeServerAddr, activeServer,
                                actorSystem.provider().getDefaultAddress());
                        activeServerAddr = activeServer;
                        heartbeatActor = actorSystem.actorSelection(getActiveHeartbeatAkkaPath(activeServerAddr));
                        instanceStatusRouter = actorSystem.actorSelection(getServerInstanceStatusRouterAkkaPath());
                        mapMasterRouter = actorSystem.actorSelection(getServerMapMasterRouterAkkaPath());
                        taskStatusRouter = actorSystem.actorSelection(getServerTaskStatusRouterAkkaPath());
                    }
                } catch (Throwable t) {
                    LOGGER.error("scheduled query active server error!", t);
                }
            }
        }, 0L, 5L, TimeUnit.SECONDS);
    }

    @Override
    public String getActiveServerAddr() {
        return activeServerAddr;
    }

    private String getServerInstanceStatusRouterAkkaPath() {
        return Constants.SERVER_AKKA_PATH_PREFIX + activeServerAddr + Constants.SERVER_AKKA_INSTANCE_STATUS_ROUTER_PATH;
    }

    private String getServerMapMasterRouterAkkaPath() {
        return Constants.SERVER_AKKA_PATH_PREFIX + activeServerAddr + Constants.SERVER_AKKA_MAP_MASTER_ROUTER_PATH;
    }

    private String getServerTaskStatusRouterAkkaPath() {
        return Constants.SERVER_AKKA_PATH_PREFIX + activeServerAddr + Constants.SERVER_AKKA_TASK_STATUS_ROUTER_PATH;
    }

    private String getActiveHeartbeatAkkaPath(String serverAddr) {
        return Constants.SERVER_AKKA_PATH_PREFIX + serverAddr + Constants.SERVER_AKKA_HEARTBEAT_PATH;
    }

    @Override
    public ActorSelection getActiveHeartBeatActor() {
        return heartbeatActor;
    }

    @Override
    public void stop() throws Exception {
        scheduledExecutorService.shutdown();
    }

    private String queryActiveServer(String domain, String groupId, String namespace, String namespaceSource) {
        String activeServer = null;
        String activeServerQueryUrl;
        if (namespace != null) {
            activeServerQueryUrl = "http://" + domain + ACTIVE_SERVER_QUERY_PATH
                    + "?groupId=" + groupId + "&namespace=" + namespace;
            if (StringUtils.isNotBlank(namespaceSource)) {
                activeServerQueryUrl += "&namespaceSource="+namespaceSource;
            }
        } else {
            activeServerQueryUrl = "http://" + domain + ACTIVE_SERVER_QUERY_PATH + "?groupId=" + groupId;
        }
        activeServerQueryUrl += "&enableScale=true";
        try {
            HttpResponse<JsonNode> jsonResponse = Unirest.get(activeServerQueryUrl).asJson();
            JSONResult jsonResult = JsonUtil.fromJson(jsonResponse.getBody().toString(), JSONResult.class);
            LOGGER.debug("queryActiveServer url={}, response={}", activeServerQueryUrl, jsonResponse.getBody().toString());
            if (jsonResult != null && jsonResult.isSuccess()) {
                if (jsonResult.getCode() == ResponseCode.GROUP_HAS_CHILD) {
                    //该应用分组开启了自动扩容并且分裂过孩子，需要解析出所有的groupIds，并注册serverDiscovery
                    ScaleGroupResult groupResult = JsonUtil.fromJson(jsonResult.getData().toString(), ScaleGroupResult.class);
                    activeServer = groupResult.getCurrentLeaderAddr();
                    //返回的分组数大于缓存，说明有新的分裂，需要新注册
                    if (CollectionUtils.isNotEmpty(groupResult.getGroupIds())) {
                        for (String childGroup : groupResult.getGroupIds()) {
                            if (!groupManager.contains(childGroup)) {
                                groupManager.startServerDiscovery(childGroup);
                                groupManager.appendGroupId(childGroup, groupId);
                            }
                        }
                    }
                } else {
                    activeServer = (String)jsonResult.getData();
                }
            }
        } catch (Throwable e) {
            LOGGER.error("query active server error, url=" + activeServerQueryUrl, e);
        }
        return activeServer;
    }


    @Override
    public ActorSelection getInstanceStatusRouter() {
        return instanceStatusRouter;
    }

    @Override
    public ActorSelection getMapMasterRouter() {
        return mapMasterRouter;
    }

    @Override
    public ActorSelection getTaskStatusRouter() {
        return taskStatusRouter;
    }

    @Deprecated
    @Override
    public List<ActorSelection> getStandbyServerHeatbeatActors() {
        return standbyServerHeatbeatActors;
    }

    @Override
    public void reset(ActorSystem actorSystem) {
        activeServerAddr = null;
        heartbeatActor = null;
        instanceStatusRouter = null;
        mapMasterRouter = null;
        taskStatusRouter = null;
        this.actorSystem = actorSystem;
        
    }
}
