package com.alibaba.schedulerx.worker.container;

import java.io.IOException;

import com.alibaba.schedulerx.common.constants.CommonConstants;
import com.alibaba.schedulerx.common.domain.InstanceStatus;
import com.alibaba.schedulerx.common.util.ConfigUtil;
import com.alibaba.schedulerx.common.util.ExceptionUtil;
import com.alibaba.schedulerx.protocol.Worker.ContainerReportTaskStatusRequest;
import com.alibaba.schedulerx.worker.SchedulerxWorker;
import com.alibaba.schedulerx.worker.batch.ContainerStatusReqHandlerPool;
import com.alibaba.schedulerx.worker.domain.JobContext;
import com.alibaba.schedulerx.worker.domain.WorkerConstants;
import com.alibaba.schedulerx.worker.log.LogFactory;
import com.alibaba.schedulerx.worker.log.Logger;
import com.alibaba.schedulerx.worker.logcollector.ClientLoggerMessage;
import com.alibaba.schedulerx.worker.logcollector.LogCollector;
import com.alibaba.schedulerx.worker.logcollector.LogCollectorFactory;
import com.alibaba.schedulerx.worker.processor.JobProcessor;
import com.alibaba.schedulerx.worker.processor.MapJobProcessor;
import com.alibaba.schedulerx.worker.processor.ProcessResult;
import com.alibaba.schedulerx.worker.processor.ProcessorFactory;
import com.alibaba.schedulerx.worker.util.JavaProcessorProfileUtil;
import com.alibaba.schedulerx.worker.util.WorkerIdGenerator;

import akka.actor.ActorSelection;
import akka.actor.Address;

/**
 *
 * @author xiaomeng.hxm
 */
public class ThreadContainer implements Runnable, Container {

    private JobContext context;
    //BaseProcessor=>JobProcessor when SimpleJobProcessor removed
    private JobProcessor jobProcessor;
    protected ContainerPool containerPool;
    protected ActorSelection masterActorSelection;
    private static final int RESULT_SIZE_MAX = CommonConstants.INSTANCE_RESULT_SIZE_MAX;
    private LogCollector logCollector = LogCollectorFactory.get();
    private static final Logger LOGGER = LogFactory.getLogger(ThreadContainer.class);

    public ThreadContainer() {}

    public ThreadContainer(JobContext context, ContainerPool containerPool) throws Exception {
        this.context = context;
        this.containerPool = containerPool;
        this.masterActorSelection = SchedulerxWorker.actorSystem.actorSelection(context.getInstanceMasterActorPath());
        if (masterActorSelection == null) {
            String errMsg = "get taskMaster akka path error, path=" + context.getInstanceMasterActorPath();
            LOGGER.error(errMsg);
            throw new IOException(errMsg);
        }
    }

    @Override
    public void run() {
        start();
    }

    @Override
    public void start() {
        containerPool.setContext(context);
        long start = System.currentTimeMillis();
        LOGGER.debug("start run container, uniqueId={}, cost={}ms", context.getUniqueId(),
                (start-context.getScheduleTime().getMillis()));

        ProcessResult result = new ProcessResult(false);
        Address address = SchedulerxWorker.actorSystem.provider().getDefaultAddress();
        String workerAddr = address.host().get() + ":" + address.port().get();
        String uniqueId = context.getUniqueId();

        try {
            reportTaskStatus(new ProcessResult(InstanceStatus.RUNNING), workerAddr);

            if ("java".equalsIgnoreCase(context.getJobType())) {
                jobProcessor = JavaProcessorProfileUtil.getJavaProcessor(context.getContent());
            } else {
                jobProcessor = ProcessorFactory.create(context.getJobType());
            }

            if (jobProcessor != null) {
                if ("java".equalsIgnoreCase(context.getJobType()) &&
                        (jobProcessor instanceof MapJobProcessor || context.getExecuteMode().equals("broadcast"))) {
                    result = jobProcessor.process(context);
                } else {
                    jobProcessor.preProcess(context);
                    result = jobProcessor.process(context);
                    jobProcessor.postProcess(context);
                }

                long end = System.currentTimeMillis();
                LOGGER.debug("container run finished, uniqueId={}, cost={}ms", uniqueId, (end-start));
                // processor执行失败打印
                if (result.getStatus() != null && result.getStatus().getValue() == InstanceStatus.FAILED.getValue()) {
                    logCollector.collect(uniqueId, ClientLoggerMessage.appendMessage(ClientLoggerMessage.JOB_PROCESSOR_EXEC_FAIL, result.getResult()));
                }
            } else {
                result = new ProcessResult(InstanceStatus.FAILED, "jobProcessor is null!");
                logCollector.collect(uniqueId, ClientLoggerMessage.appendMessage(ClientLoggerMessage.JOB_PROCESSOR_EXEC_FAIL, result.getResult()));
            }

            // map模型子任务(非根任务)失败自动重试
            if (context.getTaskMaxAttempt() > 0 && context.getTaskId() > 0 && result.getStatus().equals(InstanceStatus.FAILED)) {
                int taskAttempt = context.getTaskAttempt();
                if (taskAttempt < context.getTaskMaxAttempt()) {
                    taskAttempt++;
                    try {
                        Thread.sleep(1000 * context.getTaskAttemptInterval());
                    } catch (InterruptedException e) {
                        LOGGER.error("", e);
                    }
                    context.setTaskAttempt(taskAttempt);
                    run();
                } else {
                    reportTaskStatus(result, workerAddr);
                }
            } else {
                reportTaskStatus(result, workerAddr);
            }
        } catch (Throwable t) {
            LOGGER.error("run fail uniqueId={}", uniqueId, t);
            String errMsg = ExceptionUtil.getTrace(t);
            if (errMsg.getBytes().length > RESULT_SIZE_MAX) {
                byte[] tmp = new byte[RESULT_SIZE_MAX];
                System.arraycopy(errMsg.getBytes(), 0, tmp, 0, RESULT_SIZE_MAX);
                String fixedErrMsg = new String(tmp);
                result = new ProcessResult(InstanceStatus.FAILED, fixedErrMsg);
            } else {
                result = new ProcessResult(InstanceStatus.FAILED, errMsg);
            }
            logCollector.collect(uniqueId,ClientLoggerMessage.JOB_PROCESSOR_EXEC_FAIL, t);
            reportTaskStatus(result, workerAddr);
        } finally {
            containerPool.remove(context.getUniqueId());
            containerPool.removeContext();
        }

    }

    @Override
    public void kill() {
        LOGGER.info("kill container, jobInstanceId={}", context.getJobInstanceId());
        if (jobProcessor != null) {
            ((JobProcessor)jobProcessor).kill(context);
        }

        Address address = SchedulerxWorker.actorSystem.provider().getDefaultAddress();
        String workerAddr = address.host().get() + ":" + address.port().get();
        ContainerReportTaskStatusRequest.Builder builder = ContainerReportTaskStatusRequest.newBuilder()
                .setJobId(context.getJobId())
                .setJobInstanceId(context.getJobInstanceId())
                .setTaskId(context.getTaskId())
                .setStatus(InstanceStatus.FAILED.getValue())
                .setWorkerAddr(workerAddr)
                .setWorkerId(WorkerIdGenerator.get())
                .setResult("killed");
        if (context.getTaskName() != null) {
            builder.setTaskName(context.getTaskName());
        }
        masterActorSelection.tell(builder.build(), null);
        containerPool.remove(context.getUniqueId());
    }

    private void reportTaskStatus(ProcessResult result, String workerAddr) {
        ContainerReportTaskStatusRequest.Builder resultBuilder = ContainerReportTaskStatusRequest.newBuilder();
        resultBuilder.setJobId(context.getJobId());
        resultBuilder.setJobInstanceId(context.getJobInstanceId());
        resultBuilder.setTaskId(context.getTaskId());
        resultBuilder.setStatus(result.getStatus().getValue());
        resultBuilder.setWorkerAddr(workerAddr);
        resultBuilder.setWorkerId(WorkerIdGenerator.get());
        resultBuilder.setSerialNum(context.getSerialNum());
        if (context.getTaskName() != null) {
            resultBuilder.setTaskName(context.getTaskName());
        }
        if (result.getResult() != null) {
            resultBuilder.setResult(result.getResult());
        }
        
        boolean enableShareContainerPool = ConfigUtil.getWorkerConfig().getBoolean(WorkerConstants.SHARE_CONTAINER_POOL, false);
        boolean submitResult = false;
        if (enableShareContainerPool) {
            submitResult = ContainerStatusReqHandlerPool.INSTANCE.submitReq(0, resultBuilder.build());
        } else {
            submitResult = ContainerStatusReqHandlerPool.INSTANCE.submitReq(context.getJobInstanceId(), resultBuilder.build());
        }
        LOGGER.info("reportTaskStatus instanceId={} submitResult={}, processResult={}", context.getUniqueId(), 
                submitResult, result);
        if (!submitResult) {
            masterActorSelection.tell(resultBuilder.build(), null);
        }
    }

    public JobContext getContext() {
        return context;
    }

    public void setContext(JobContext context) {
        this.context = context;
    }

}
