package com.alibaba.schedulerx.worker.batch;

import java.util.List;
import java.util.Map;
import java.util.Map.Entry;

import com.alibaba.schedulerx.common.util.ConfigUtil;
import com.alibaba.schedulerx.protocol.Worker.ContainerBatchReportTaskStatuesRequest;
import com.alibaba.schedulerx.protocol.Worker.ContainerReportTaskStatusRequest;
import com.alibaba.schedulerx.protocol.Worker.TaskStatusInfo;
import com.alibaba.schedulerx.worker.SchedulerxWorker;
import com.alibaba.schedulerx.worker.domain.WorkerConstants;
import com.alibaba.schedulerx.worker.log.LogFactory;
import com.alibaba.schedulerx.worker.log.Logger;

import com.google.common.collect.Lists;
import com.google.common.collect.Maps;

/**
 * batch report container task status to task master
 * @author yanxun on 2019/1/6.
 */
public class ContainerStatusReqHandler<T> extends BaseReqHandler<T> {
    private String taskMasterAkkaPath;
    private boolean enableShareContainerPool = ConfigUtil.getWorkerConfig().getBoolean(WorkerConstants.SHARE_CONTAINER_POOL, false);
    private static final Logger LOGGER = LogFactory.getLogger(ContainerStatusReqHandler.class);

    @Override
    public void process(long jobInstanceId, List<T> reqs, String workerAddr) {
        batchProcessSvc.submit(new BatchStatuesReportRunnable(jobInstanceId, (List<ContainerReportTaskStatusRequest>)reqs));
    }

    public ContainerStatusReqHandler(long jobInstanceId, int coreBatchThreadNum, int maxBatchThreadNum,
                                     int batchSize, ReqQueue<T> queue, String taskMasterAkkaPath) {
        super(jobInstanceId, coreBatchThreadNum, maxBatchThreadNum, batchSize, queue,
            "Schedulerx-Container-Batch-Statuses-Process-Thread-", "Schedulerx-Container-Batch-Statues-Retrieve-Thread-");
        this.taskMasterAkkaPath = taskMasterAkkaPath;
        defaultSleepMs = 10;
    }

    private class BatchStatuesReportRunnable implements Runnable {
        private long jobInstanceId;
        private List<ContainerReportTaskStatusRequest> statues;
        BatchStatuesReportRunnable(long jobInstanceId, List<ContainerReportTaskStatusRequest> reqs) {
            this.jobInstanceId = jobInstanceId;
            this.statues = reqs;
        }

        @Override
        public void run() {
            try {
                if (enableShareContainerPool) {
                    //如果开启共享线程池，statues可能会有多个jobInstanceId，需要先split成不同的list
                    Map<Long, List<ContainerReportTaskStatusRequest>> taskStatusRequestMap = Maps.newHashMap();
                    for (ContainerReportTaskStatusRequest req : statues) {
                        long jobInstanceId = req.getJobInstanceId();
                        if (taskStatusRequestMap.containsKey(jobInstanceId)) {
                            taskStatusRequestMap.get(jobInstanceId).add(req);
                        } else {
                            List<ContainerReportTaskStatusRequest> reqsByInstance = Lists.newArrayList(req);
                            taskStatusRequestMap.put(jobInstanceId, reqsByInstance);
                        }
                    }
                    
                    //针对不同的jobInstanceId，构造batchStatusRequests
                    for (Entry<Long, List<ContainerReportTaskStatusRequest>> entry : taskStatusRequestMap.entrySet()) {
                        long jobInstanceId = entry.getKey();
                        List<TaskStatusInfo> taskStatuses = Lists.newArrayList();
                        for (ContainerReportTaskStatusRequest req : entry.getValue()) {
                            TaskStatusInfo.Builder builder = TaskStatusInfo.newBuilder()
                                    .setTaskId(req.getTaskId())
                                    .setStatus(req.getStatus());
                            if (req.hasTaskName()) {
                                builder.setTaskName(req.getTaskName());
                            }
                            if (req.hasResult()) {
                                builder.setResult(req.getResult());
                            }
                            taskStatuses.add(builder.build());
                        }
                        
                        ContainerReportTaskStatusRequest taskStatusRequest = entry.getValue().get(0);
                        ContainerBatchReportTaskStatuesRequest request = ContainerBatchReportTaskStatuesRequest.newBuilder()
                                .setJobId(taskStatusRequest.getJobId())
                                .setJobInstanceId(jobInstanceId)
                                .addAllTaskStatues(taskStatuses)
                                .setTaskMasterAkkaPath(taskMasterAkkaPath)
                                .setWorkerAddr(taskStatusRequest.getWorkerAddr())
                                .setWorkerId(taskStatusRequest.getWorkerId())
                                .setSerialNum(taskStatusRequest.getSerialNum())
                                .build();
                        SchedulerxWorker.AtLeastDeliveryRoutingActor.tell(request, null);
                        LOGGER.info("jobInstanceId={} batch report status={} to task master, size:{}", jobInstanceId, 
                                taskStatusRequest.getStatus(), statues.size());
                    }
                } else {
                    List<TaskStatusInfo> taskStatuses = Lists.newArrayList();
                    // some attrs are duplicated in all reqs, for example: workAddr, workerId, jobId, jobInstanceId, taskMasterPath
                    // get first one used for all reqs.
                    ContainerReportTaskStatusRequest taskStatusRequest = statues.get(0);
                    TaskStatusInfo.Builder builder;
                    for (ContainerReportTaskStatusRequest req : statues) {
                        builder = TaskStatusInfo.newBuilder().setTaskId(req.getTaskId()).setStatus(req.getStatus());
                        if (req.hasTaskName()) {
                            builder.setTaskName(req.getTaskName());
                        }
                        if (req.hasResult()) {
                            builder.setResult(req.getResult());
                        }
                        taskStatuses.add(builder.build());
                    }
                    ContainerBatchReportTaskStatuesRequest request = ContainerBatchReportTaskStatuesRequest.newBuilder()
                        .setJobId(taskStatusRequest.getJobId())
                        .setJobInstanceId(taskStatusRequest.getJobInstanceId())
                        .addAllTaskStatues(taskStatuses)
                        .setTaskMasterAkkaPath(taskMasterAkkaPath)
                        .setWorkerAddr(taskStatusRequest.getWorkerAddr())
                        .setWorkerId(taskStatusRequest.getWorkerId())
                        .setSerialNum(taskStatusRequest.getSerialNum())
                        .build();
                    SchedulerxWorker.AtLeastDeliveryRoutingActor.tell(request, null);
                    LOGGER.info("jobInstanceId={} batch report status={} to task master, size:{}", jobInstanceId, taskStatusRequest.getStatus(), statues.size());
                }
                
            } catch (Throwable e) {
                LOGGER.error(e);
            } finally {
                activeRunnableNum.decrementAndGet();
            }
        }
    }

    public String getTaskMasterAkkaPath() {
        return taskMasterAkkaPath;
    }
}
