package com.alibaba.schedulerx.sdk;

import java.util.HashMap;
import java.util.Map;

import com.alibaba.schedulerx.common.constants.CommonConstants;
import com.alibaba.schedulerx.common.domain.ResponseCode;
import com.alibaba.schedulerx.common.exception.RequestCheckException;
import com.alibaba.schedulerx.common.sdk.common.ParamValidate;
import com.alibaba.schedulerx.common.sdk.common.SchedulerXResult;
import com.alibaba.schedulerx.common.sdk.request.BaseRequest;
import com.alibaba.schedulerx.common.sdk.request.CreateAppGroupRequest;
import com.alibaba.schedulerx.common.sdk.request.CreateNamespaceRequest;
import com.alibaba.schedulerx.common.sdk.response.BaseResponse;
import com.alibaba.schedulerx.common.util.SignatureUtil;
import com.alibaba.schedulerx.sdk.http.HttpClientUtil;
import com.alibaba.schedulerx.worker.log.LogFactory;
import com.alibaba.schedulerx.worker.log.Logger;

import org.apache.commons.lang.StringUtils;

/**
 * sdk入口
 *
 * @author 陆昆
 * @create 2018-12-16 上午11:44
 **/
public class OpenApiClient {
    private static final Logger logger = LogFactory.getLogger("openapi");
    /**
     * schedulerx应用管理界面查看用户身份校验(必填)
     */
    private String appKey;

    /**
     * 分组id应用管理界面查看(必填)
     */
    private String groupId;

    /**
     * 发送到对应的域名(可选)默认根据机器所在的环境填写
     * 日常：
     * 预发：
     * 线上：
     */
    private String domain;

    /**
     * 操作人方便记录查询(可选)
     */
    @Deprecated
    private String user;
    
    /**
     * 用户id
     */
    private String userId;
    
    /**
     * 用户名
     */
    private String userName;

    /**
     * 命名空间特定环境使用
     */
    private String namespace;

    private String namespaceSource;

    public OpenApiClient(String appKey, String groupId, String domain) {
        this.appKey = appKey;
        this.groupId = groupId;
        // 判断是否填写http或者https
        if (domain.contains("http")) {
            this.domain = domain;
        } else {
            this.domain = "http://" + domain;
        }
    }

    @Deprecated
    public OpenApiClient(String appKey, String groupId, String domain, String user) {
        this(appKey, groupId, domain);
        this.user = user;
    }
    
    public OpenApiClient(String domain, String groupId, String appKey, String userId, String userName) {
        this(appKey, groupId, domain);
        this.userId = userId;
        this.userName = userName;
    }

    public OpenApiClient(String domain, String user) {
        this.domain = domain;
        this.user = user;
        if (domain.contains("http")) {
            this.domain = domain;
        } else {
            this.domain = "http://" + domain;
        }
    }

    public <T extends BaseResponse> SchedulerXResult<T> getResponse(BaseRequest<T> request) throws Exception {
        // 填充初始化参数
        if (groupId != null) {
            request.setGroupId(groupId);
        }
        if (namespace != null) {
            request.setNamespace(namespace);
        }
        if (namespaceSource != null) {
            request.setNamespaceSource(namespaceSource);
        }
        
        // 操作人填充
        if (userId != null && userName != null) {
            request.getParameterMap().put("user", userName + " " + userId);
        } else {
            request.getParameterMap().put("user", StringUtils.isBlank(user) ? "openapi" : user);
        }
        // 必填参数校验
        validateCommonParam();
        // header填充
        Map<String, String> headers = genHeaders(request);
        // 参数注解校验
        ParamValidate.doValidator(request);
        // 部分请求预处理
        request.preRun();
        logger.info("getResponse url: {} request params : {}", request.getUrlPath(),
            request.getParameterMap().toString());
        // http请求
        return HttpClientUtil.getResponse(request, this.getDomain(), headers);
    }

    /**
     * 填充header
     */
    private Map<String, String> genHeaders(BaseRequest request) throws Exception {
        Map<String, String> headers = new HashMap<>();
        // 非必填参数
        if (StringUtils.isNotBlank(namespace)) {
            headers.put(CommonConstants.NAMESPACE_KEY_HEADER, namespace);
            if (StringUtils.isNotBlank(namespaceSource)) {
                headers.put(CommonConstants.NAMESPACE_SOURCE_HEADER, namespaceSource);
            }
        }
        String timeNow = Long.toString(System.currentTimeMillis());
        // 时间
        headers.put(CommonConstants.TIME_STAMP_HEADER, timeNow);
        // 签名
        if (groupId != null && appKey != null) {
            headers.put(CommonConstants.SIGNATURE_HEADER, SignatureUtil.geneHmacSHA1Encrypt(timeNow, groupId, appKey));
        }
        // groupId
        headers.put(CommonConstants.GROUPID_HEADER, groupId);
        
        // 是否创建命名空间
        if (request instanceof CreateNamespaceRequest) {
            headers.put(CommonConstants.CREATE_NAMESPACE_HEADER, "true");
        } else {
            headers.put(CommonConstants.CREATE_NAMESPACE_HEADER, "false");
        }
        
        // 是否创建分组
        if (request instanceof CreateAppGroupRequest) {
            headers.put(CommonConstants.CREATE_GROUP_HEADER, "true");
        } else {
            headers.put(CommonConstants.CREATE_GROUP_HEADER, "false");
        }
        return headers;
    }

    /**
     * 公共参数校验
     */
    private void validateCommonParam() {
        if (StringUtils.isBlank(domain)) {
            throw new RequestCheckException(ResponseCode.OPEN_API_REQUEST_ERROR,
                "domain can not be blank");
        }
    }

    public String getAppKey() {
        return appKey;
    }

    public void setAppKey(String appKey) {
        this.appKey = appKey;
    }

    public String getGroupId() {
        return groupId;
    }

    public void setGroupId(String groupId) {
        this.groupId = groupId;
    }

    public String getDomain() {
        return domain;
    }

    public void setDomain(String domain) {
        this.domain = domain;
    }

    public void setNamespace(String namespace) {
        this.namespace = namespace;
    }

    public String getNamespace() {
        return namespace;
    }

    public String getNamespaceSource() {
        return namespaceSource;
    }

    public void setNamespaceSource(String namespaceSource) {
        this.namespaceSource = namespaceSource;
    }

    public String getUserId() {
        return userId;
    }

    public void setUserId(String userId) {
        this.userId = userId;
    }

    public String getUserName() {
        return userName;
    }

    public void setUserName(String userName) {
        this.userName = userName;
    }
    
}
