/*
 * Decompiled with CFR 0.152.
 */
package org.springframework.data.redis.core;

import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import org.aopalliance.aop.Advice;
import org.aopalliance.intercept.MethodInterceptor;
import org.aopalliance.intercept.MethodInvocation;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.springframework.aop.RawTargetAccess;
import org.springframework.aop.framework.ProxyFactory;
import org.springframework.dao.DataAccessException;
import org.springframework.data.redis.connection.RedisConnection;
import org.springframework.data.redis.connection.RedisConnectionFactory;
import org.springframework.data.redis.core.RedisCommand;
import org.springframework.lang.Nullable;
import org.springframework.transaction.support.ResourceHolderSupport;
import org.springframework.transaction.support.TransactionSynchronization;
import org.springframework.transaction.support.TransactionSynchronizationManager;
import org.springframework.util.Assert;

public abstract class RedisConnectionUtils {
    private static final Log log = LogFactory.getLog(RedisConnectionUtils.class);

    public static RedisConnection bindConnection(RedisConnectionFactory factory) {
        return RedisConnectionUtils.doGetConnection(factory, true, true, false);
    }

    public static RedisConnection bindConnection(RedisConnectionFactory factory, boolean transactionSupport) {
        return RedisConnectionUtils.doGetConnection(factory, true, true, transactionSupport);
    }

    public static RedisConnection getConnection(RedisConnectionFactory factory) {
        return RedisConnectionUtils.getConnection(factory, false);
    }

    public static RedisConnection getConnection(RedisConnectionFactory factory, boolean transactionSupport) {
        return RedisConnectionUtils.doGetConnection(factory, true, false, transactionSupport);
    }

    public static RedisConnection doGetConnection(RedisConnectionFactory factory, boolean allowCreate, boolean bind, boolean transactionSupport) {
        boolean bindSynchronization;
        Assert.notNull((Object)factory, (String)"No RedisConnectionFactory specified");
        RedisConnectionHolder conHolder = (RedisConnectionHolder)((Object)TransactionSynchronizationManager.getResource((Object)factory));
        if (conHolder != null && (conHolder.hasConnection() || conHolder.isSynchronizedWithTransaction())) {
            conHolder.requested();
            if (!conHolder.hasConnection()) {
                log.debug((Object)"Fetching resumed Redis Connection from RedisConnectionFactory");
                conHolder.setConnection(RedisConnectionUtils.fetchConnection(factory));
            }
            return conHolder.getRequiredConnection();
        }
        if (!allowCreate) {
            throw new IllegalArgumentException("No connection found and allowCreate = false");
        }
        log.debug((Object)"Fetching Redis Connection from RedisConnectionFactory");
        RedisConnection connection = RedisConnectionUtils.fetchConnection(factory);
        boolean bl = bindSynchronization = TransactionSynchronizationManager.isActualTransactionActive() && transactionSupport;
        if (bind || bindSynchronization) {
            if (bindSynchronization && RedisConnectionUtils.isActualNonReadonlyTransactionActive()) {
                connection = RedisConnectionUtils.createConnectionSplittingProxy(connection, factory);
            }
            try {
                RedisConnectionHolder holderToUse = conHolder;
                if (holderToUse == null) {
                    holderToUse = new RedisConnectionHolder(connection);
                } else {
                    holderToUse.setConnection(connection);
                }
                holderToUse.requested();
                if (bindSynchronization) {
                    RedisConnectionUtils.potentiallyRegisterTransactionSynchronisation(holderToUse, factory);
                }
                if (holderToUse != conHolder) {
                    TransactionSynchronizationManager.bindResource((Object)factory, (Object)((Object)holderToUse));
                }
            }
            catch (RuntimeException ex) {
                RedisConnectionUtils.releaseConnection(connection, factory);
                throw ex;
            }
            return connection;
        }
        return connection;
    }

    private static RedisConnection fetchConnection(RedisConnectionFactory factory) {
        return factory.getConnection();
    }

    private static void potentiallyRegisterTransactionSynchronisation(RedisConnectionHolder connHolder, RedisConnectionFactory factory) {
        if (!connHolder.isTransactionActive()) {
            connHolder.setTransactionActive(true);
            connHolder.setSynchronizedWithTransaction(true);
            connHolder.requested();
            RedisConnection conn = connHolder.getRequiredConnection();
            boolean readOnly = TransactionSynchronizationManager.isCurrentTransactionReadOnly();
            if (!readOnly) {
                conn.multi();
            }
            TransactionSynchronizationManager.registerSynchronization((TransactionSynchronization)new RedisTransactionSynchronizer(connHolder, conn, factory, readOnly));
        }
    }

    private static boolean isActualNonReadonlyTransactionActive() {
        return TransactionSynchronizationManager.isActualTransactionActive() && !TransactionSynchronizationManager.isCurrentTransactionReadOnly();
    }

    private static RedisConnection createConnectionSplittingProxy(RedisConnection connection, RedisConnectionFactory factory) {
        ProxyFactory proxyFactory = new ProxyFactory((Object)connection);
        proxyFactory.addAdvice((Advice)new ConnectionSplittingInterceptor(factory));
        proxyFactory.addInterface(RedisConnectionProxy.class);
        return (RedisConnection)RedisConnection.class.cast(proxyFactory.getProxy());
    }

    public static void releaseConnection(@Nullable RedisConnection conn, RedisConnectionFactory factory) {
        if (conn == null) {
            return;
        }
        RedisConnectionHolder conHolder = (RedisConnectionHolder)((Object)TransactionSynchronizationManager.getResource((Object)factory));
        if (conHolder != null) {
            if (conHolder.isTransactionActive()) {
                if (RedisConnectionUtils.connectionEquals(conHolder, conn)) {
                    if (log.isDebugEnabled()) {
                        log.debug((Object)"RedisConnection will be closed when transaction finished.");
                    }
                    conHolder.released();
                }
                return;
            }
            RedisConnectionUtils.unbindConnection(factory);
            return;
        }
        RedisConnectionUtils.doCloseConnection(conn);
    }

    @Deprecated
    public static void releaseConnection(@Nullable RedisConnection conn, RedisConnectionFactory factory, boolean transactionSupport) {
        RedisConnectionUtils.releaseConnection(conn, factory);
    }

    private static boolean connectionEquals(RedisConnectionHolder conHolder, RedisConnection passedInCon) {
        if (!conHolder.hasConnection()) {
            return false;
        }
        RedisConnection heldCon = conHolder.getRequiredConnection();
        return heldCon.equals(passedInCon) || RedisConnectionUtils.getTargetConnection(heldCon).equals(passedInCon);
    }

    private static RedisConnection getTargetConnection(RedisConnection con) {
        RedisConnection conToUse = con;
        while (conToUse instanceof RedisConnectionProxy) {
            conToUse = ((RedisConnectionProxy)conToUse).getTargetConnection();
        }
        return conToUse;
    }

    public static void unbindConnection(RedisConnectionFactory factory) {
        RedisConnectionHolder conHolder = (RedisConnectionHolder)((Object)TransactionSynchronizationManager.getResource((Object)factory));
        if (conHolder == null) {
            return;
        }
        if (log.isDebugEnabled()) {
            log.debug((Object)"Unbinding Redis Connection.");
        }
        if (conHolder.isTransactionActive()) {
            if (log.isDebugEnabled()) {
                log.debug((Object)"Redis Connection will be closed when outer transaction finished.");
            }
        } else {
            RedisConnection connection = conHolder.getConnection();
            conHolder.released();
            if (!conHolder.isOpen()) {
                TransactionSynchronizationManager.unbindResourceIfPossible((Object)factory);
                RedisConnectionUtils.doCloseConnection(connection);
            }
        }
    }

    public static boolean isConnectionTransactional(RedisConnection conn, RedisConnectionFactory connFactory) {
        Assert.notNull((Object)connFactory, (String)"No RedisConnectionFactory specified");
        RedisConnectionHolder connHolder = (RedisConnectionHolder)((Object)TransactionSynchronizationManager.getResource((Object)connFactory));
        return connHolder != null && RedisConnectionUtils.connectionEquals(connHolder, conn);
    }

    private static void doCloseConnection(@Nullable RedisConnection connection) {
        if (connection == null) {
            return;
        }
        if (log.isDebugEnabled()) {
            log.debug((Object)"Closing Redis Connection.");
        }
        try {
            connection.close();
        }
        catch (DataAccessException ex) {
            log.debug((Object)"Could not close Redis Connection", (Throwable)ex);
        }
        catch (Throwable ex) {
            log.debug((Object)"Unexpected exception on closing Redis Connection", ex);
        }
    }

    static interface RedisConnectionProxy
    extends RedisConnection,
    RawTargetAccess {
        public RedisConnection getTargetConnection();
    }

    private static class RedisConnectionHolder
    extends ResourceHolderSupport {
        @Nullable
        private RedisConnection connection;
        private boolean transactionActive = false;

        public RedisConnectionHolder(RedisConnection connection) {
            this.connection = connection;
        }

        protected boolean hasConnection() {
            return this.connection != null;
        }

        @Nullable
        public RedisConnection getConnection() {
            return this.connection;
        }

        public RedisConnection getRequiredConnection() {
            RedisConnection connection = this.getConnection();
            if (connection == null) {
                throw new IllegalStateException("No active RedisConnection");
            }
            return connection;
        }

        protected void setConnection(@Nullable RedisConnection connection) {
            this.connection = connection;
        }

        protected void setTransactionActive(boolean transactionActive) {
            this.transactionActive = transactionActive;
        }

        protected boolean isTransactionActive() {
            return this.transactionActive;
        }

        public void released() {
            super.released();
            if (!this.isOpen()) {
                this.setConnection(null);
            }
        }

        public void clear() {
            super.clear();
            this.transactionActive = false;
        }
    }

    static class ConnectionSplittingInterceptor
    implements MethodInterceptor {
        private final RedisConnectionFactory factory;

        public ConnectionSplittingInterceptor(RedisConnectionFactory factory) {
            this.factory = factory;
        }

        public Object invoke(MethodInvocation invocation) throws Throwable {
            return this.intercept(invocation.getThis(), invocation.getMethod(), invocation.getArguments());
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        public Object intercept(Object obj, Method method, Object[] args) throws Throwable {
            if (method.getName().equals("getTargetConnection")) {
                return obj;
            }
            RedisCommand commandToExecute = RedisCommand.failsafeCommandLookup(method.getName());
            if (this.isPotentiallyThreadBoundCommand(commandToExecute)) {
                if (log.isDebugEnabled()) {
                    log.debug((Object)String.format("Invoke '%s' on bound connection", method.getName()));
                }
                return this.invoke(method, obj, args);
            }
            if (log.isDebugEnabled()) {
                log.debug((Object)String.format("Invoke '%s' on unbound connection", method.getName()));
            }
            RedisConnection connection = this.factory.getConnection();
            try {
                Object object = this.invoke(method, connection, args);
                return object;
            }
            finally {
                if (!connection.isClosed()) {
                    RedisConnectionUtils.doCloseConnection(connection);
                }
            }
        }

        private Object invoke(Method method, Object target, Object[] args) throws Throwable {
            try {
                return method.invoke(target, args);
            }
            catch (InvocationTargetException e) {
                throw e.getCause();
            }
        }

        private boolean isPotentiallyThreadBoundCommand(RedisCommand command) {
            return RedisCommand.UNKNOWN.equals((Object)command) || !command.isReadonly();
        }
    }

    private static class RedisTransactionSynchronizer
    implements TransactionSynchronization {
        private final RedisConnectionHolder connHolder;
        private final RedisConnection connection;
        private final RedisConnectionFactory factory;
        private final boolean readOnly;

        RedisTransactionSynchronizer(RedisConnectionHolder connHolder, RedisConnection connection, RedisConnectionFactory factory, boolean readOnly) {
            this.connHolder = connHolder;
            this.connection = connection;
            this.factory = factory;
            this.readOnly = readOnly;
        }

        /*
         * Enabled force condition propagation
         * Lifted jumps to return sites
         */
        public void afterCompletion(int status) {
            try {
                if (this.readOnly) return;
                switch (status) {
                    case 0: {
                        this.connection.exec();
                        return;
                    }
                    default: {
                        this.connection.discard();
                        return;
                    }
                }
            }
            finally {
                if (log.isDebugEnabled()) {
                    log.debug((Object)("Closing bound connection after transaction completed with " + status));
                }
                this.connHolder.setTransactionActive(false);
                RedisConnectionUtils.doCloseConnection(this.connection);
                TransactionSynchronizationManager.unbindResource((Object)this.factory);
                this.connHolder.reset();
            }
        }
    }
}

