/*
 * Copyright (c) 2002-2018 "Neo4j,"
 * Neo4j Sweden AB [http://neo4j.com]
 *
 * This file is part of Neo4j.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.neo4j.driver.internal.messaging;

import java.io.IOException;
import java.time.Instant;
import java.time.LocalDate;
import java.time.LocalDateTime;
import java.time.LocalTime;
import java.time.OffsetTime;
import java.time.ZoneId;
import java.time.ZoneOffset;
import java.time.ZonedDateTime;

import org.neo4j.driver.internal.InternalPoint2D;
import org.neo4j.driver.internal.InternalPoint3D;
import org.neo4j.driver.internal.packstream.PackInput;
import org.neo4j.driver.internal.packstream.PackOutput;
import org.neo4j.driver.internal.types.TypeConstructor;
import org.neo4j.driver.internal.value.InternalValue;
import org.neo4j.driver.v1.Value;
import org.neo4j.driver.v1.types.IsoDuration;
import org.neo4j.driver.v1.types.Point;

import static java.time.ZoneOffset.UTC;
import static org.neo4j.driver.v1.Values.isoDuration;
import static org.neo4j.driver.v1.Values.point;
import static org.neo4j.driver.v1.Values.value;

public class PackStreamMessageFormatV2 extends PackStreamMessageFormatV1
{
    private static final byte DATE = 'D';
    private static final int DATE_STRUCT_SIZE = 1;

    private static final byte TIME = 'T';
    private static final int TIME_STRUCT_SIZE = 2;

    private static final byte LOCAL_TIME = 't';
    private static final int LOCAL_TIME_STRUCT_SIZE = 1;

    private static final byte LOCAL_DATE_TIME = 'd';
    private static final int LOCAL_DATE_TIME_STRUCT_SIZE = 2;

    private static final byte DATE_TIME_WITH_ZONE_OFFSET = 'F';
    private static final byte DATE_TIME_WITH_ZONE_ID = 'f';
    private static final int DATE_TIME_STRUCT_SIZE = 3;

    private static final byte DURATION = 'E';
    private static final int DURATION_TIME_STRUCT_SIZE = 4;

    private static final byte POINT_2D_STRUCT_TYPE = 'X';
    private static final int POINT_2D_STRUCT_SIZE = 3;

    private static final byte POINT_3D_STRUCT_TYPE = 'Y';
    private static final int POINT_3D_STRUCT_SIZE = 4;

    @Override
    public MessageFormat.Writer newWriter( PackOutput output, boolean byteArraySupportEnabled )
    {
        if ( !byteArraySupportEnabled )
        {
            throw new IllegalArgumentException( "Bolt V2 should support byte arrays" );
        }
        return new WriterV2( output );
    }

    @Override
    public MessageFormat.Reader newReader( PackInput input )
    {
        return new ReaderV2( input );
    }

    static class WriterV2 extends WriterV1
    {
        WriterV2( PackOutput output )
        {
            super( output, true );
        }

        @Override
        void packInternalValue( InternalValue value ) throws IOException
        {
            TypeConstructor typeConstructor = value.typeConstructor();
            switch ( typeConstructor )
            {
            case DATE:
                packDate( value.asLocalDate() );
                break;
            case TIME:
                packTime( value.asOffsetTime() );
                break;
            case LOCAL_TIME:
                packLocalTime( value.asLocalTime() );
                break;
            case LOCAL_DATE_TIME:
                packLocalDateTime( value.asLocalDateTime() );
                break;
            case DATE_TIME:
                packZonedDateTime( value.asZonedDateTime() );
                break;
            case DURATION:
                packDuration( value.asIsoDuration() );
                break;
            case POINT:
                packPoint( value.asPoint() );
                break;
            default:
                super.packInternalValue( value );
            }
        }

        private void packDate( LocalDate localDate ) throws IOException
        {
            packer.packStructHeader( DATE_STRUCT_SIZE, DATE );
            packer.pack( localDate.toEpochDay() );
        }

        private void packTime( OffsetTime offsetTime ) throws IOException
        {
            long nanoOfDayLocal = offsetTime.toLocalTime().toNanoOfDay();
            int offsetSeconds = offsetTime.getOffset().getTotalSeconds();

            packer.packStructHeader( TIME_STRUCT_SIZE, TIME );
            packer.pack( nanoOfDayLocal );
            packer.pack( offsetSeconds );
        }

        private void packLocalTime( LocalTime localTime ) throws IOException
        {
            packer.packStructHeader( LOCAL_TIME_STRUCT_SIZE, LOCAL_TIME );
            packer.pack( localTime.toNanoOfDay() );
        }

        private void packLocalDateTime( LocalDateTime localDateTime ) throws IOException
        {
            long epochSecondUtc = localDateTime.toEpochSecond( UTC );
            int nano = localDateTime.getNano();

            packer.packStructHeader( LOCAL_DATE_TIME_STRUCT_SIZE, LOCAL_DATE_TIME );
            packer.pack( epochSecondUtc );
            packer.pack( nano );
        }

        private void packZonedDateTime( ZonedDateTime zonedDateTime ) throws IOException
        {
            long epochSecondLocal = zonedDateTime.toLocalDateTime().toEpochSecond( UTC );
            int nano = zonedDateTime.getNano();

            ZoneId zone = zonedDateTime.getZone();
            if ( zone instanceof ZoneOffset )
            {
                int offsetSeconds = ((ZoneOffset) zone).getTotalSeconds();

                packer.packStructHeader( DATE_TIME_STRUCT_SIZE, DATE_TIME_WITH_ZONE_OFFSET );
                packer.pack( epochSecondLocal );
                packer.pack( nano );
                packer.pack( offsetSeconds );
            }
            else
            {
                String zoneId = zone.getId();

                packer.packStructHeader( DATE_TIME_STRUCT_SIZE, DATE_TIME_WITH_ZONE_ID );
                packer.pack( epochSecondLocal );
                packer.pack( nano );
                packer.pack( zoneId );
            }
        }

        private void packDuration( IsoDuration duration ) throws IOException
        {
            packer.packStructHeader( DURATION_TIME_STRUCT_SIZE, DURATION );
            packer.pack( duration.months() );
            packer.pack( duration.days() );
            packer.pack( duration.seconds() );
            packer.pack( duration.nanoseconds() );
        }

        private void packPoint( Point point ) throws IOException
        {
            if ( point instanceof InternalPoint2D )
            {
                packPoint2D( point );
            }
            else if ( point instanceof InternalPoint3D )
            {
                packPoint3D( point );
            }
            else
            {
                throw new IOException( String.format( "Unknown type: type: %s, value: %s",  point.getClass(), point.toString() ) );
            }
        }

        private void packPoint2D ( Point point ) throws IOException
        {
            packer.packStructHeader( POINT_2D_STRUCT_SIZE, POINT_2D_STRUCT_TYPE );
            packer.pack( point.srid() );
            packer.pack( point.x() );
            packer.pack( point.y() );
        }

        private void packPoint3D( Point point ) throws IOException
        {
            packer.packStructHeader( POINT_3D_STRUCT_SIZE, POINT_3D_STRUCT_TYPE );
            packer.pack( point.srid() );
            packer.pack( point.x() );
            packer.pack( point.y() );
            packer.pack( point.z() );
        }
    }

    static class ReaderV2 extends ReaderV1
    {
        ReaderV2( PackInput input )
        {
            super( input );
        }

        @Override
        Value unpackStruct( long size, byte type ) throws IOException
        {
            switch ( type )
            {
            case DATE:
                ensureCorrectStructSize( TypeConstructor.DATE, DATE_STRUCT_SIZE, size );
                return unpackDate();
            case TIME:
                ensureCorrectStructSize( TypeConstructor.TIME, TIME_STRUCT_SIZE, size );
                return unpackTime();
            case LOCAL_TIME:
                ensureCorrectStructSize( TypeConstructor.LOCAL_TIME, LOCAL_TIME_STRUCT_SIZE, size );
                return unpackLocalTime();
            case LOCAL_DATE_TIME:
                ensureCorrectStructSize( TypeConstructor.LOCAL_DATE_TIME, LOCAL_DATE_TIME_STRUCT_SIZE, size );
                return unpackLocalDateTime();
            case DATE_TIME_WITH_ZONE_OFFSET:
                ensureCorrectStructSize( TypeConstructor.DATE_TIME, DATE_TIME_STRUCT_SIZE, size );
                return unpackDateTimeWithZoneOffset();
            case DATE_TIME_WITH_ZONE_ID:
                ensureCorrectStructSize( TypeConstructor.DATE_TIME, DATE_TIME_STRUCT_SIZE, size );
                return unpackDateTimeWithZoneId();
            case DURATION:
                ensureCorrectStructSize( TypeConstructor.DURATION, DURATION_TIME_STRUCT_SIZE, size );
                return unpackDuration();
            case POINT_2D_STRUCT_TYPE:
                ensureCorrectStructSize( TypeConstructor.POINT, POINT_2D_STRUCT_SIZE, size );
                return unpackPoint2D();
            case POINT_3D_STRUCT_TYPE:
                ensureCorrectStructSize( TypeConstructor.POINT, POINT_3D_STRUCT_SIZE, size );
                return unpackPoint3D();
            default:
                return super.unpackStruct( size, type );
            }
        }

        private Value unpackDate() throws IOException
        {
            long epochDay = unpacker.unpackLong();
            return value( LocalDate.ofEpochDay( epochDay ) );
        }

        private Value unpackTime() throws IOException
        {
            long nanoOfDayLocal = unpacker.unpackLong();
            int offsetSeconds = Math.toIntExact( unpacker.unpackLong() );

            LocalTime localTime = LocalTime.ofNanoOfDay( nanoOfDayLocal );
            ZoneOffset offset = ZoneOffset.ofTotalSeconds( offsetSeconds );
            return value( OffsetTime.of( localTime, offset ) );
        }

        private Value unpackLocalTime() throws IOException
        {
            long nanoOfDayLocal = unpacker.unpackLong();
            return value( LocalTime.ofNanoOfDay( nanoOfDayLocal ) );
        }

        private Value unpackLocalDateTime() throws IOException
        {
            long epochSecondUtc = unpacker.unpackLong();
            int nano = Math.toIntExact( unpacker.unpackLong() );
            return value( LocalDateTime.ofEpochSecond( epochSecondUtc, nano, UTC ) );
        }

        private Value unpackDateTimeWithZoneOffset() throws IOException
        {
            long epochSecondLocal = unpacker.unpackLong();
            int nano = Math.toIntExact( unpacker.unpackLong() );
            int offsetSeconds = Math.toIntExact( unpacker.unpackLong() );
            return value( newZonedDateTime( epochSecondLocal, nano, ZoneOffset.ofTotalSeconds( offsetSeconds ) ) );
        }

        private Value unpackDateTimeWithZoneId() throws IOException
        {
            long epochSecondLocal = unpacker.unpackLong();
            int nano = Math.toIntExact( unpacker.unpackLong() );
            String zoneIdString = unpacker.unpackString();
            return value( newZonedDateTime( epochSecondLocal, nano, ZoneId.of( zoneIdString ) ) );
        }

        private Value unpackDuration() throws IOException
        {
            long months = unpacker.unpackLong();
            long days = unpacker.unpackLong();
            long seconds = unpacker.unpackLong();
            int nanoseconds = Math.toIntExact( unpacker.unpackLong() );
            return isoDuration( months, days, seconds, nanoseconds );
        }

        private Value unpackPoint2D() throws IOException
        {
            int srid = Math.toIntExact( unpacker.unpackLong() );
            double x = unpacker.unpackDouble();
            double y = unpacker.unpackDouble();
            return point( srid, x, y );
        }

        private Value unpackPoint3D() throws IOException
        {
            int srid = Math.toIntExact( unpacker.unpackLong() );
            double x = unpacker.unpackDouble();
            double y = unpacker.unpackDouble();
            double z = unpacker.unpackDouble();
            return point( srid, x, y, z );
        }

        private static ZonedDateTime newZonedDateTime( long epochSecondLocal, long nano, ZoneId zoneId )
        {
            Instant instant = Instant.ofEpochSecond( epochSecondLocal, nano );
            LocalDateTime localDateTime = LocalDateTime.ofInstant( instant, UTC );
            return ZonedDateTime.of( localDateTime, zoneId );
        }
    }
}
