package com.ifourthwall.message.sms;

import com.aliyuncs.CommonRequest;
import com.aliyuncs.CommonResponse;
import com.aliyuncs.DefaultAcsClient;
import com.aliyuncs.IAcsClient;
import com.aliyuncs.exceptions.ClientException;
import com.aliyuncs.http.MethodType;
import com.aliyuncs.profile.DefaultProfile;
import com.ifourthwall.message.sms.config.SMSProperties;
import com.ifourthwall.common.utils.JSONUtils;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;

import java.util.List;
import java.util.Map;

/**
 * @Author: hao.dai
 * @Date: 2019/12/27 16:08
 * @Description:
 */
@Slf4j
public class SendSMSService {
    public static final String EN_COUNTRY_CODE = "+1";
    private Map<String, List<SMSProperties>> smsPropertiesMap;
    private List<SMSProperties> defaultSMSProperties;

    public SendSMSService(Map<String, List<SMSProperties>> smsPropertiesMap, List<SMSProperties> defaultSMSProperties) {
        this.smsPropertiesMap = smsPropertiesMap;
        this.defaultSMSProperties = defaultSMSProperties;
    }

    public boolean sendSMS(String phoneNos, String index, String messageJson) {
        return sendSMS(phoneNos, index, messageJson, null);
    }

    public boolean sendSMS(String phoneNos, String index, String messageJson, SendSMSCallBack sendSMSCallBack) {
        SMSProperties smsProperties = chooseSMSProperties(phoneNos, index);
        String finalPhoneNo = phoneNos.replace("+", "");

        DefaultProfile profile = DefaultProfile.getProfile(smsProperties.getRegionId(), smsProperties.getAccessKeyId(),
                smsProperties.getSecret());
        IAcsClient client = new DefaultAcsClient(profile);
        CommonRequest request = new CommonRequest();
        request.setMethod(MethodType.POST);
        request.setDomain(smsProperties.getDomain());
        request.setVersion(smsProperties.getVersion());
        request.setAction(smsProperties.getAction());
        request.putQueryParameter("RegionId", smsProperties.getRegionId());
        //阿里云区号+86需要去掉"+"
        request.putQueryParameter("PhoneNumbers", finalPhoneNo);
        request.putQueryParameter("SignName", new String(smsProperties.getSignature().getBytes()));
        request.putQueryParameter("TemplateCode", smsProperties.getTemplateCode());
        request.putQueryParameter("TemplateParam", messageJson);

        try {
            if (sendSMSCallBack != null) {
                sendSMSCallBack.beforeSend(client, smsProperties, messageJson);
            }
            CommonResponse response = client.getCommonResponse(request);
            Map<String, String> data = JSONUtils.jacksonFromJson(response.getData(), Map.class);

            if (data != null && data.get("Code").equals("OK")) {
                log.info("发送信息成功ok：" + finalPhoneNo + " message:" + messageJson + " return:" + response.getData());
                if (sendSMSCallBack != null) {
                    sendSMSCallBack.onSuccess(data);
                }
            } else {
                log.error("发送信息失败：" + finalPhoneNo + " message:" + messageJson + " return:" + response.getData());
                if (sendSMSCallBack != null) {
                    sendSMSCallBack.onFailed(data);
                }
            }
            return true;
        } catch (ClientException e) {
            log.error("发送信息失败：" + finalPhoneNo + " message:" + messageJson);
            log.error("ErrCode:" + e.getErrCode());
            log.error("ErrMsg:" + e.getErrMsg());
            log.error("RequestId:" + e.getRequestId());
            if (sendSMSCallBack != null) {
                sendSMSCallBack.onException(e);
            }
            return false;
        }
    }

    /***
     * 根据手机号码区号获取对应的模版
     * 1.接收号码格式为国际区号+手机号码，如“+8613712345678”。
     * 2.若不填写国家区号，则会使用默认短信模版（国内模版）
     * 3.国际号码和手机号码之间不要加0。如在越南拨打当地号码是04987654321（04是当地拨打的区号），国家码+84与当地号码04987654321的0是需要去掉的,拨打国际号码则变为是+844987654321
     */
    private SMSProperties chooseSMSProperties(String phoneNumbers, String index) {
        if (phoneNumbers != null && !phoneNumbers.equals("")) {
            //选择第一个带区号的电话作为模版 比如 +1 + 86 + 81的电话号码列表，返回+1所对应的模版
            String phoneNoArray[] = phoneNumbers.split(",");
            if (phoneNoArray != null) {
                for (String phoneNo : phoneNoArray) {

                    for (Map.Entry<String, List<SMSProperties>> template : smsPropertiesMap.entrySet()) {

                        if (!phoneNo.startsWith(template.getKey())) {
                            continue;
                        }
                        List<SMSProperties> smsPropertiesList = template.getValue();
                        SMSProperties smsProperties = getMacthedSMSProperties(smsPropertiesList, index);
                        if (smsProperties != null) {
                            return smsProperties;
                        }
                    }
                }
                //若不存在+81 +86 +1 的但存在其他区号，则选择+1对应的模版，英语
                if (phoneNumbers.contains("+") && smsPropertiesMap.containsKey(EN_COUNTRY_CODE)) {
                    List<SMSProperties> smsPropertiesList = smsPropertiesMap.get(EN_COUNTRY_CODE);
                    SMSProperties smsProperties = getMacthedSMSProperties(smsPropertiesList, index);
                    if (smsProperties != null) {
                        return smsProperties;
                    }
                }
            }
            //获取默认模板
            for (SMSProperties defaultSMSProperty : defaultSMSProperties) {
                if (StringUtils.equals(defaultSMSProperty.getIndex(), index)) {
                    return defaultSMSProperty;
                }
            }
        }
        throw new IllegalArgumentException("SMS发送:index未指定或" + index + "下未指定默认模板");
    }

    private SMSProperties getMacthedSMSProperties(List<SMSProperties> smsPropertiesList, String index) {
        SMSProperties defaultSMSProperties = null;
        SMSProperties matchedSMSProperties = null;
        for (SMSProperties smsProperties : smsPropertiesList) {
            //获取默认模板
            if (smsProperties.isAsDefaultTemplate()) {
                defaultSMSProperties = smsProperties;
            }
            //获取匹配的模板
            if (StringUtils.equals(smsProperties.getIndex(), index)) {
                matchedSMSProperties = smsProperties;
            }
        }
        //未匹配到返回默认模板
        if (matchedSMSProperties != null) {
            return matchedSMSProperties;
        } else {
            return defaultSMSProperties;
        }
    }
}
